function [x,y,rainrate] = rainfieldx(nt,monthplot,dayplot,hourplot)
%
%  This script calculates the distribution of surface rain rate, in mm/hr,
%  for a given storm at a specified time.
%
%  nt is the track number of the storm, monthplot is the month (0-12), day
%  plot is the day, and hourplot is the GMT time.
%
%  x and y are vectors containing the longitudes and latitudes of the grid
%
%  rainrate is the surface rainfall rate, in mm/hr, at each grid point.
%
%  The grid resolution and dimensions can be set automatically or
%  specified in the windfield map section of params.m.
%
%       December 2011
%       Copyright 2011 WindRiskTech L.L.C.
%       Revised December 2020 to remove cos(latitude) factor from hy and to
%       allow for higher resolution topography
%------------------------------------------------------------------------
%
params    %  Load parameters
%
load temp vstore vsestore latstore longstore rmstore rmsestore ut vt
load temp T600store u850store v850store monthstore daystore hourstore
load sorted %#ok<LOAD>
%
if strcmpi(topores,'high')
    load bathymetry_high
else
    load bathymetry
end    
[ntopo,~]=size(bathy);
topores=360/ntopo;
toporesi=1/topores;
%    
[nrm,mrm]=size(rmstore);
rfac=magfac.*(1+zeros(nrm,mrm));
%
% Randomize radii of maximum wind
%
if strcmp(randfac,'y')
    rfac=magfac.*outer(nrm,mrm);
end
%
pifac=acos(-1)/180;
knotfac=1852./3600;
sfac=1./(topores.*60.*1852);
%
[~, jmaxd]=min(vstore,[],2);
jmaxd=jmaxd-1;
jmaxd=transpose(jmaxd);
%
clear timev V
dum=zeros(1,jmaxd(nt));
V=[dum+2000; monthstore(nt,1:jmaxd(nt)); daystore(nt,1:jmaxd(nt));...
    hourstore(nt,1:jmaxd(nt)); dum; dum];
V=transpose(V);
timev=datenum(V);
timeplot=datenum([2000 monthplot dayplot hourplot 0 0]);
[~,jplot]=min(max(timeplot-timev,0));
jplot=max(jplot,2);
jplot=min(jplot,399);
%
latstorm=latstore(nt,jplot-1:jplot+1);
longstorm=longstore(nt,jplot-1:jplot+1);
vstorm=vstore(nt,jplot-1:jplot+1);
rmstorm=rfac(nt,jplot-1:jplot+1).*(rmstore(nt,jplot-1:jplot+1));
vsestorm=vsestore(nt,jplot-1:jplot+1);
rmsestorm=rfac(nt,jplot-1:jplot+1).*(rmsestore(nt,jplot-1:jplot+1));
utstorm=ut(nt,jplot-1:jplot+1);
vtstorm=vt(nt,jplot-1:jplot+1);
ush=zeros(1,3);
vsh=ush;
if exist('u850store','var')
    vdrift=1.5.*3600./1852;
    vdrift=vdrift.*latstore(1,1)./(abs(latstore(1,1))+1e-8);
    u850storm=u850store(nt,jplot-1:jplot+1);
    v850storm=v850store(nt,jplot-1:jplot+1);
    ush(1,:)=5.*knotfac.*(utstorm-u850storm);
    vsh(1,:)=5.*knotfac.*(vtstorm-vdrift.*cos(pifac.*latstorm)-v850storm);
end 
%
if char(bound) == char('auto')
    bxmin=floor(longstorm(1,2)-deltax);
    bxmax=ceil(longstorm(1,2)+deltax);
    bymin=floor(latstorm(1,2)-deltay);
    bymax=ceil(latstorm(1,2)+deltay);
end
%
x=bxmin:dellatlong:bxmax;
y=bymin:dellatlong:bymax;
sx=max(size(x));
sy=max(size(y));
%
%  This statement added because in pointwshort it is assumed that if sx=sy
%  POI's are a vector rather than a grid.
if sx == sy
    sx=sx+1;
    x=[x bxmax+dellatlong];
end  
%
h=zeros(sx,sy);
hx=zeros(sx,sy);
hy=zeros(sx,sy);
m_to_mm=1000;
rowa_over_rowl=0.00117;
%
bathy=max(bathy,-1);
dhdx=sfac*(circshift(bathy,-1,1)-circshift(bathy,0,1));
dhdy=sfac*(circshift(bathy,-1,2)-circshift(bathy,0,2));
%
for i=1:sx
    plong=x(i);
    for j=1:sy
        plat=y(j);
        %
        ib=1+floor(toporesi.*plong);
        ibp=ib+1;
        if ibp > ntopo
            ibp=ibp-ntopo;
        end    
        ibs=1+floor(toporesi.*plong-0.5);
        ibsp=ibs+1;
        plongs=plong;
        if ibs < 0
            ibs=ntopo;
            plongs=plong+360;
        end    
        if ibsp > ntopo
            ibsp=ibsp-ntopo;
        end
        jb=1+floor(toporesi.*(plat+90));
        jbs=1+floor(toporesi.*(plat+90)-0.5);
        b1=bathy(ib,jb);
        b2=bathy(ib,jb+1);
        b3=bathy(ibp,jb);
        b4=bathy(ibp,jb+1);
        dely=1+toporesi.*(plat+90)-jb;
        delx=1+toporesi.*plong-ib;
        d1=(1.-delx).*(1.-dely);
        d2=dely.*(1.-delx);
        d3=delx.*(1.-dely);
        d4=delx.*dely;
        h(i,j)=exp(d1.*log(b1+11)+d2.*log(b2+11)+d3.*log(b3+11)+d4.*log(b4+11))-11;
        %
        b1=dhdx(ibs,jbs);
        b2=dhdx(ibs,jbs+1);
        b3=dhdx(ibsp,jbs);
        b4=dhdx(ibsp,jbs+1);
        dely=0.5+toporesi.*(plat+90)-jbs;
        delx=0.5+toporesi.*plongs-ibs;
        d1=(1.-delx).*(1.-dely);
        d2=dely.*(1.-delx);
        d3=delx.*(1.-dely);
        d4=delx.*dely;
        hx(i,j)=(b1.*d1+b2.*d2+b3.*d3+b4.*d4)./cos(pifac.*plat);
        %
        b1=dhdy(ibs,jbs);
        b2=dhdy(ibs,jbs+1);
        b3=dhdy(ibsp,jbs);
        b4=dhdy(ibsp,jbs+1);
        hy(i,j)=(b1.*d1+b2.*d2+b3.*d3+b4.*d4);    
        %
    end
end  
clear bathy
if exist('T600store','var')
    clear q900 q600
    [ q900, ~ ] = qs900b( T600store(nt,jplot),vstore(nt,jplot) );
end    
w = pointwfield(latstorm,longstorm,vstorm,rmstorm,vsestorm,rmsestorm, ...
    utstorm,vtstorm,ush,vsh,y,x,h,hx,hy);
temp=eprecip.*m_to_mm.*3600.*rowa_over_rowl.*q900.*max(w(1,2,:,:)-wrad,0);
rainrate(:,:)=temp(1,1,:,:);
rainrate=transpose(rainrate);
%
end
