function [p] = VtoP(vstore, latstore, r0)
%
% Estimates central pressure based on circular 10m wind (vstore; kts),
% latitude (latstore; degrees), and outer radius (r0; km). All three inputs
% may be matrices or scalars. Note that vstore should be non-negative in
% both hemispheres. We recommend that the median (or scalar) value of r0 be
% 500 km. Setting the parameter CF to 'y' (with r0 constant) sets the outer
% radius of each event to a random draw from the log-normal distribution of
% outer radii, with a mean of log(r0), based on the distribution of
% historical storms found by Chavas and Emanuel (GRL; 2010.). Estimating
% central pressure this way gives a better fit to extended best-track data
% than does the provided matrix pstore.
%
% Output: p is the matrix of central pressures (hPa)
%
% Copyright WindRiskTech, LLC   January, 2024
%--------------------------------------------------------------------------
p0=1012; % Ambient pressure (hPa); 
a=1;     % Form factor for Coriolis term
b=1.3;   % Form factor for centripetal term
CF='n';  % Use random draw from Chavas function, with r0 the mean (scalar) radius?
%--------------------------------------------------------------------------
RTs=86100;               % Product of gas constant and surface temperature
RTsi=1/RTs;
%
v=1852*vstore/3600;      % Convert to m/s
[n,m]=size(v);
%
f=1.4544e-4*sin(pi*abs(latstore)/180);
if strcmpi(CF,'y') && max(size(r0)) == 1
    r=lognrnd(log(r0),0.35,[1,n]); % Assign different outer radii to each track
    r=repmat(r',[1,m]);            % Repeat outer radius along all of each track
else
    r=r0;
end    
r=1000.*r;                         % Convert to meters
p=p0*exp(RTsi*(-b*v.^2-a*f.*v.*r));
%
end