function cols=m_colmap(nme,m,ncol)
% M_COLMAP  Useful colormaps
%   M_COLMAP(NAME) returns an M-by-3 matrix containing the NAME colormap
%   where NAME is one of:
%     'jet' : a perceptually uniform variation of the JET colormap. It 
%             contains the multiple colours which make JET useful, while 
%             avoiding the weird highlighting especially around yellow and 
%             cyan. The colors begin with dark blue, range through shades 
%             of blue, green, orange and red, and ends with dark red. 
%     'blue' : a perceptually useful blue shading (good for bathymetry)
%     'green' : a perceptually useful green shading
%     'diverging' : a blue/red diverging colormap
%     'odv'  : an isoluminant map
%     'land' : a topographic height (green-brown-white) shading
%     'water' : blue shading for water (goes with 'land').
%     'gland' : a topographic height  shading with more green
%     'bland' : a topographic height shading with browns only.
%
%   and M is the same length as the current figure's colormap. If no 
%   figure exists, the length of the default colormap is used. The
%   length can be explicitly specified with M_COLMAP(NAME,M).
%
%   M_COLMAP('demo') demonstrates the colormaps.
%
%   M_COLMAP(NAME,'step') returns a 256-color map in which colours are
%   perceptually bunched into 16 separate colours. This is useful 
%   if you want to see "edges" in what would be an otherwise
%   smooth gradation (i.e. approaching the look of contouring). 
%
%   M_COLMAP(NAME,'demo') Gives a demo of this behavior
%  
%   M_COLMAP(NAME,'step',M) bunches into M colours.
%
%   The jet and odv colourmaps were designed using the CET Perceptually 
%   Uniform Colour Maps toolbox at peterkovesi.com/projects/colourmaps/
%
%   The blue, green, and diverging colormaps are from CBREWER, a matlab
%   implementation of colours described at http://colorbrewer2.org.
%
%   The land and gland colormaps are derived from the ETOPO1 water/land 
%   colormap at http://soliton.vm.bytemark.co.uk/pub/cpt-city/ngdc/index.html
%
%   See also PARULA, HSV, HOT, PINK, FLAG, COLORMAP, RGBPLOT.

% R Pawlowicz Nov/2017

% names
if nargin==0
    m_colmap('get');
    return;
end

switch lower(nme(1:3))
    case 'jet'
        nme='jet';
    case 'blu'
        nme='blue';
    case 'gre'
        nme='green';
    case 'div'
        nme='diverging';
    case 'odv'
        nme='odv';
    case 'lan'
        nme='land';
    case 'wat'
        nme='water';
    case 'gla'
        nme='gland';
    case 'bla'
        nme='bland';
    case 'get'
        fprintf('Colormaps include:\n');
        fprintf('   ''jet'' {,M|,''step''{,M}}\n');
        fprintf('   ''blue'' {,M|,''step''{,M}}\n');
        fprintf('   ''green'' {,M|,''step''{,M}}\n');
        fprintf('   ''diverging'' {,M|,''step''{,M}}\n');  
        fprintf('   ''land'' {,M|,''step''{,M}}\n');
        fprintf('   ''gland'' {,M|,''step''{,M}}\n');
        fprintf('   ''bland'' {,M|,''step''{,M}}\n');
        fprintf('   ''water'' {,M|,''step''{,M}}\n');
    case 'dem'
        colmap_demo;
        return
    otherwise
     error(['map: ' mfilename ':invalidMapname'],...
            ' First argument must be an valid colormap name ');
end


if nargin < 2
   f = gcf;
   m = size(get(f,'Colormap'),1);
   ncol=m;
else 
  if isnumeric(m)
    ncol=m;
  else
    switch m(1:3)
      case 'dem'   % demo
         clf;
         subplot(2,1,1);
         imagesc(peaks(10*10));
         title('Peaks function displayed with IMAGESC');
         subplot(2,1,2);
         imagesc(1:256);
	     cols=m_colmap(nme,'step');
         colormap(cols);
         title(['The colormap used above: m\_colmap(''' nme ''',256,16)']);
	     if nargout==0
	       clear cols
	     end  
         return;

      case 'ste'    % step
         m=256;
         if nargin<3
	        ncol=16;
         end  
      otherwise
         error(['map: ' mfilename ':invalidOption'],...
            ' Second argument must be an valid option');
     end
   end    
end
   
% Whatever length colormap we have
 
if ncol<m/2
  xx=[0:(m-1)]'/(m-1)*pi*ncol;
  yy=cumsum(cos(xx).^4 + 1/30);
  yy=yy-yy(1);
  yy=yy/max(yy)*255;
else
  yy=[0:(m-1)]'/(m-1)*255;
end
  
if nargout>0
  [r,basecols]=mycols(nme);
  cols=interp1(r,basecols,yy);
end



end



function [r,Data]=mycols(nme)
%
% Returns raw colourmap - 0<=r<=255 and
% [0 0 0]<=Data(row,:)<=[1 1 1]
%

r=[0:255]; % Default for most of these.
switch nme
    case 'jet'
        Data=[...
           0.0000000000000000e+00   0.0000000000000000e+00   8.2097987032366326e-01
           0.0000000000000000e+00   0.0000000000000000e+00   8.2810185708388151e-01
           0.0000000000000000e+00   1.6244801768825715e-02   8.3523804428475601e-01
           0.0000000000000000e+00   3.3036471648334995e-02   8.4238547984011569e-01
           0.0000000000000000e+00   4.9815313296183217e-02   8.4954008831091266e-01
           0.0000000000000000e+00   6.4029974334975218e-02   8.5670244416752706e-01
           0.0000000000000000e+00   7.6535959975665882e-02   8.6388438002720569e-01
           0.0000000000000000e+00   8.8013865699010582e-02   8.7105469052628581e-01
           0.0000000000000000e+00   9.8600980176800038e-02   8.7821143148088110e-01
           0.0000000000000000e+00   1.0855337679557225e-01   8.8532422388838095e-01
           0.0000000000000000e+00   1.1812539206874099e-01   8.9236038456129052e-01
           0.0000000000000000e+00   1.2726308783402634e-01   8.9926170553603024e-01
           0.0000000000000000e+00   1.3619917896297423e-01   9.0595230153199524e-01
           0.0000000000000000e+00   1.4506118969622983e-01   9.1232172813169876e-01
           0.0000000000000000e+00   1.5377890552725290e-01   9.1823409334055117e-01
           0.0000000000000000e+00   1.6261365170233669e-01   9.2351427566801991e-01
           0.0000000000000000e+00   1.7154074948381501e-01   9.2794646225220667e-01
           0.0000000000000000e+00   1.8070256379609848e-01   9.3131899478139557e-01
           0.0000000000000000e+00   1.9017887289112076e-01   9.3338241072484596e-01
           0.0000000000000000e+00   1.9997783192436316e-01   9.3392154621153756e-01
           0.0000000000000000e+00   2.1019119971968461e-01   9.3274045531274641e-01
           0.0000000000000000e+00   2.2078830806041727e-01   9.2970178242713375e-01
           0.0000000000000000e+00   2.3170788397910427e-01   9.2472675872455312e-01
           0.0000000000000000e+00   2.4291058720581157e-01   9.1783217933659011e-01
           0.0000000000000000e+00   2.5439689340691901e-01   9.0907966847559551e-01
           0.0000000000000000e+00   2.6601511242910086e-01   8.9863138701299472e-01
           0.0000000000000000e+00   2.7770789341482360e-01   8.8668327712843464e-01
           0.0000000000000000e+00   2.8940651531291417e-01   8.7345282146153225e-01
           0.0000000000000000e+00   3.0102041216268027e-01   8.5918692717013723e-01
           0.0000000000000000e+00   3.1252327536427388e-01   8.4411199405573478e-01
           0.0000000000000000e+00   3.2386802405602533e-01   8.2843647500968565e-01
           0.0000000000000000e+00   3.3502534309709658e-01   8.1233252376445897e-01
           0.0000000000000000e+00   3.4591820537992940e-01   7.9594701247201749e-01
           0.0000000000000000e+00   3.5661427072219648e-01   7.7939361987056155e-01
           0.0000000000000000e+00   3.6703710422430275e-01   7.6275539017719729e-01
           0.0000000000000000e+00   3.7721017332291257e-01   7.4610487594974018e-01
           0.0000000000000000e+00   3.8709592071744631e-01   7.2949277318103267e-01
           0.0000000000000000e+00   3.9670474982054660e-01   7.1295673554612449e-01
           0.0000000000000000e+00   4.0602438327051393e-01   6.9655675508496329e-01
           0.0000000000000000e+00   4.1500561550145126e-01   6.8030864681305458e-01
           0.0000000000000000e+00   4.2366289307317656e-01   6.6426240828904592e-01
           0.0000000000000000e+00   4.3200712706984007e-01   6.4842386975272537e-01
           3.0566524560783279e-03   4.3999127354957085e-01   6.3282182663250508e-01
           2.1259594640010666e-02   4.4764881469817519e-01   6.1745638962714788e-01
           4.3803836689370011e-02   4.5501957945750093e-01   6.0233557663170700e-01
           6.5847269982244869e-02   4.6208402302425822e-01   5.8744496819586389e-01
           8.5714142185715644e-02   4.6890583204144093e-01   5.7277998204506675e-01
           1.0409596887009070e-01   4.7550636110461003e-01   5.5828074688552098e-01
           1.2102697913139572e-01   4.8192127879442975e-01   5.4395369747328215e-01
           1.3667902142111943e-01   4.8820071402585424e-01   5.2974025041805628e-01
           1.5103696859797716e-01   4.9437177951608591e-01   5.1559526287877988e-01
           1.6416735480159775e-01   5.0046436145743556e-01   5.0151677895326252e-01
           1.7599785165409093e-01   5.0651659371728996e-01   4.8743987594925903e-01
           1.8673494649110914e-01   5.1254773876555138e-01   4.7333098265162471e-01
           1.9635984308719795e-01   5.1857739467224218e-01   4.5920475411903505e-01
           2.0490590950992249e-01   5.2459707667461242e-01   4.4499829451551892e-01
           2.1249481521562941e-01   5.3063533990020706e-01   4.3070141183233479e-01
           2.1917765060054359e-01   5.3669591908749759e-01   4.1629180215797296e-01
           2.2499535088875952e-01   5.4276707337012853e-01   4.0174506555606909e-01
           2.3004736186449742e-01   5.4886421494692972e-01   3.8704858786501528e-01
           2.3439364402268012e-01   5.5497461212998789e-01   3.7217181537750266e-01
           2.3802214916377087e-01   5.6111021864810351e-01   3.5711075761778077e-01
           2.4102958080802567e-01   5.6727120617849991e-01   3.4180313298247084e-01
           2.4347078965167188e-01   5.7343697984137976e-01   3.2624718187968538e-01
           2.4541755424164602e-01   5.7961519583221810e-01   3.1046646770536129e-01
           2.4695087697744367e-01   5.8578978746082422e-01   2.9436976897198003e-01
           2.4816602364067705e-01   5.9195396468447525e-01   2.7803594615233906e-01
           2.4914804420757949e-01   5.9808477862815501e-01   2.6142686912908059e-01
           2.5009122010144075e-01   6.0419223727339377e-01   2.4457986428005363e-01
           2.5125107794115892e-01   6.1022050248824800e-01   2.2760242777985751e-01
           2.5277963649326868e-01   6.1617179084321716e-01   2.1050609513494345e-01
           2.5499887511783337e-01   6.2200150973129531e-01   1.9345024424320145e-01
           2.5811871983643009e-01   6.2770454035851242e-01   1.7653075332522819e-01
           2.6246816164275255e-01   6.3323695072608899e-01   1.5996210212986198e-01
           2.6818346039674890e-01   6.3858125645704489e-01   1.4398097898551054e-01
           2.7543338752629509e-01   6.4372114150893245e-01   1.2870986574286711e-01
           2.8418213538041015e-01   6.4865628006181242e-01   1.1444933871183166e-01
           2.9441479687067434e-01   6.5336988668170226e-01   1.0152426951505689e-01
           3.0599728766863188e-01   6.5788824014397052e-01   9.0139191413144706e-02
           3.1866282501793303e-01   6.6220359310488208e-01   8.0483895706916511e-02
           3.3224320620395048e-01   6.6633263204236381e-01   7.2801598535599379e-02
           3.4646179112792114e-01   6.7032401486586635e-01   6.7067497770581408e-02
           3.6113631412522007e-01   6.7416768477092770e-01   6.3249077315340047e-02
           3.7608731018537811e-01   6.7791233106441862e-01   6.0980464884529913e-02
           3.9114649219215752e-01   6.8155555404442747e-01   6.0237646826049381e-02
           4.0620969841971383e-01   6.8514312914868336e-01   6.0434850173806676e-02
           4.2119208206707631e-01   6.8867051135431134e-01   6.1317869943769421e-02
           4.3607730241284837e-01   6.9215167952165702e-01   6.2899876768280863e-02
           4.5080272445331049e-01   6.9559278562020677e-01   6.4630034822273294e-02
           4.6536143503241884e-01   6.9900108657586113e-01   6.6611409445883923e-02
           4.7977927019121297e-01   7.0239032638575571e-01   6.8765586458024650e-02
           4.9402278124389287e-01   7.0575587274235263e-01   7.0874113597661562e-02
           5.0812209701947120e-01   7.0910213329145999e-01   7.3065156643943877e-02
           5.2205818272473403e-01   7.1241552273829678e-01   7.5181298203090122e-02
           5.3588169773612182e-01   7.1572588892233568e-01   7.7316510344089567e-02
           5.4958796386661235e-01   7.1900198771359247e-01   7.9498911575721087e-02
           5.6318325188735974e-01   7.2225882977126032e-01   8.1739922275486848e-02
           5.7667734958705708e-01   7.2550816943346963e-01   8.3886170589407089e-02
           5.9006715802606269e-01   7.2872236431771908e-01   8.5969093205613856e-02
           6.0339198534874716e-01   7.3192594564488933e-01   8.8191004728472841e-02
           6.1663360653820753e-01   7.3510592591557200e-01   9.0348464852867502e-02
           6.2979877564545061e-01   7.3826041744638315e-01   9.2484236233912193e-02
           6.4290502541913341e-01   7.4139762708091694e-01   9.4695676765631190e-02
           6.5595456455707568e-01   7.4451903129559338e-01   9.6766064389575565e-02
           6.6894100563615921e-01   7.4762140253119092e-01   9.8966110102230720e-02
           6.8187256965709520e-01   7.5069894456381092e-01   1.0111568171360964e-01
           6.9476451769134873e-01   7.5375241811299143e-01   1.0331287444459691e-01
           7.0761384378249492e-01   7.5678894789262396e-01   1.0543392059916874e-01
           7.2042165068418740e-01   7.5980213993578027e-01   1.0760350744563843e-01
           7.3319470176764145e-01   7.6279840902831653e-01   1.0981215158679503e-01
           7.4592828571528091e-01   7.6577879845079466e-01   1.1194689692155048e-01
           7.5863468570665482e-01   7.6872670099528584e-01   1.1407090775631480e-01
           7.7131848568430994e-01   7.7166810405759800e-01   1.1627716014608953e-01
           7.8398678935532840e-01   7.7456683356005351e-01   1.1844264905380968e-01
           7.9662697582417896e-01   7.7744081794020037e-01   1.2054078396447886e-01
           8.0926185607475876e-01   7.8025300319044533e-01   1.2266783407386730e-01
           8.2187414124312763e-01   7.8300130137588675e-01   1.2480676238195154e-01
           8.3446602223939059e-01   7.8565722796567439e-01   1.2687749576058471e-01
           8.4702437066066527e-01   7.8816898161099214e-01   1.2890567635476541e-01
           8.5954289047780741e-01   7.9049204962457431e-01   1.3088474793982860e-01
           8.7197831670600356e-01   7.9255502154840052e-01   1.3269172868341109e-01
           8.8427738654617694e-01   7.9427245270485536e-01   1.3436638425443168e-01
           8.9637053729861804e-01   7.9553909023023628e-01   1.3576481705392310e-01
           9.0816444779814065e-01   7.9626169503294353e-01   1.3697659943588600e-01
           9.1953673575996886e-01   7.9633432352513089e-01   1.3786713706219822e-01
           9.3037259899096236e-01   7.9566413293269556e-01   1.3835047336237846e-01
           9.4053209111364600e-01   7.9418180027718388e-01   1.3842916138252423e-01
           9.4989207994255664e-01   7.9182887326844242e-01   1.3808153617074001e-01
           9.5836294828986246e-01   7.8861031080564803e-01   1.3725685332614224e-01
           9.6586462349906821e-01   7.8454305319884909e-01   1.3594823160952096e-01
           9.7239039139483674e-01   7.7967621023720923e-01   1.3431101877433596e-01
           9.7794177385612990e-01   7.7410205347271077e-01   1.3221480879967804e-01
           9.8259333615419153e-01   7.6791778909022512e-01   1.2980073335970169e-01
           9.8642494090379840e-01   7.6122992421425550e-01   1.2702534915073754e-01
           9.8955166954763185e-01   7.5413730388263700e-01   1.2406071795849168e-01
           9.9208992758455905e-01   7.4672824419867512e-01   1.2089567778081263e-01
           9.9415891855407701e-01   7.3909488972694637e-01   1.1760517777184400e-01
           9.9586234078238622e-01   7.3127941939636465e-01   1.1416775566548833e-01
           9.9729483003018982e-01   7.2335091637232807e-01   1.1070869560896780e-01
           9.9852727658269325e-01   7.1534184353124297e-01   1.0714334894572426e-01
           9.9960957379964188e-01   7.0726043228880653e-01   1.0352445558761095e-01
           1.0000000000000000e+00   6.9912890339600753e-01   9.9829372782125531e-02
           1.0000000000000000e+00   6.9095883736804042e-01   9.6116559516242178e-02
           1.0000000000000000e+00   6.8275705650119767e-01   9.2417489373060394e-02
           1.0000000000000000e+00   6.7452426257956222e-01   8.8677593797862289e-02
           1.0000000000000000e+00   6.6625666540404749e-01   8.4887313721726754e-02
           1.0000000000000000e+00   6.5796365276303015e-01   8.1068177807262484e-02
           1.0000000000000000e+00   6.4962701740088724e-01   7.7151213817244532e-02
           1.0000000000000000e+00   6.4124528876170073e-01   7.3351998210193614e-02
           1.0000000000000000e+00   6.3283767086881249e-01   6.9416379806024719e-02
           1.0000000000000000e+00   6.2438475544545868e-01   6.5331601005328144e-02
           1.0000000000000000e+00   6.1588280808086227e-01   6.1211652569017221e-02
           1.0000000000000000e+00   6.0734668246046875e-01   5.7123569446539284e-02
           1.0000000000000000e+00   5.9874800807184747e-01   5.2865554268786509e-02
           1.0000000000000000e+00   5.9010246493802909e-01   4.8647547440398876e-02
           1.0000000000000000e+00   5.8140951034192445e-01   4.4198570670319334e-02
           1.0000000000000000e+00   5.7266585316082430e-01   3.9705280914958743e-02
           1.0000000000000000e+00   5.6385731706157016e-01   3.5191257367171103e-02
           1.0000000000000000e+00   5.5497816055012050e-01   3.0647747322206934e-02
           1.0000000000000000e+00   5.4604761486237297e-01   2.6461611736228074e-02
           1.0000000000000000e+00   5.3704122502628560e-01   2.2436402898803214e-02
           1.0000000000000000e+00   5.2795094033373646e-01   1.8569728931739027e-02
           1.0000000000000000e+00   5.1880030560715795e-01   1.4859141284721791e-02
           1.0000000000000000e+00   5.0955197815230335e-01   1.1196012834395055e-02
           1.0000000000000000e+00   5.0021693769032671e-01   7.7056436600708390e-03
           1.0000000000000000e+00   4.9080757314831269e-01   4.4527213011970471e-03
           1.0000000000000000e+00   4.8127050430505153e-01   1.3486232596699801e-03
           1.0000000000000000e+00   4.7165053711191246e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.6190083189547609e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.5204021136716938e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.4204343506591337e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.3190627002582327e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.2159591143411812e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.1114619470166724e-01   0.0000000000000000e+00
           1.0000000000000000e+00   4.0051916733978549e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.8967477462351169e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.7863908422433407e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.6737229102939645e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.5588805687874975e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.4412420334830296e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.3213896185541525e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.1988349723243259e-01   0.0000000000000000e+00
           1.0000000000000000e+00   3.0737131121967348e-01   0.0000000000000000e+00
           1.0000000000000000e+00   2.9466234283445780e-01   0.0000000000000000e+00
           1.0000000000000000e+00   2.8180431610272688e-01   0.0000000000000000e+00
           1.0000000000000000e+00   2.6885346259089427e-01   0.0000000000000000e+00
           9.9971380921896635e-01   2.5588279168968908e-01   0.0000000000000000e+00
           9.9745107484299911e-01   2.4306303891550010e-01   0.0000000000000000e+00
           9.9471175718486304e-01   2.3053240966922203e-01   0.0000000000000000e+00
           9.9144050807482531e-01   2.1845350104652131e-01   0.0000000000000000e+00
           9.8759742405304762e-01   2.0697738741569063e-01   0.0000000000000000e+00
           9.8316166027396978e-01   1.9624582890323888e-01   0.0000000000000000e+00
           9.7814117536301937e-01   1.8638373147524231e-01   0.0000000000000000e+00
           9.7256691186559741e-01   1.7750982585337141e-01   0.0000000000000000e+00
           9.6646763333593055e-01   1.6960615229744733e-01   0.0000000000000000e+00
           9.5991667718773344e-01   1.6265283853733181e-01   0.0000000000000000e+00
           9.5297568794473220e-01   1.5659747318781603e-01   0.0000000000000000e+00
           9.4571233760129314e-01   1.5128687901353635e-01   0.0000000000000000e+00
           9.3820118122169338e-01   1.4664168011978534e-01   0.0000000000000000e+00
           9.3050268815754777e-01   1.4253427062145158e-01   0.0000000000000000e+00
           9.2266803031005806e-01   1.3878427919273950e-01   0.0000000000000000e+00
           9.1474108307638347e-01   1.3532555938839283e-01   0.0000000000000000e+00
           9.0675852417667746e-01   1.3206323417870042e-01   0.0000000000000000e+00
           8.9873962287707232e-01   1.2889180676634279e-01   0.0000000000000000e+00
           8.9070644004606214e-01   1.2586340008396713e-01   0.0000000000000000e+00
           8.8266708081748724e-01   1.2278481340036466e-01   0.0000000000000000e+00
           8.7463547859087032e-01   1.1980508899179866e-01   0.0000000000000000e+00
           8.6661131238848954e-01   1.1682153274204797e-01   0.0000000000000000e+00
           8.5859901159723606e-01   1.1382680416595528e-01   0.0000000000000000e+00
           8.5060093722026298e-01   1.1083435058870025e-01   0.0000000000000000e+00
           8.4262210493452183e-01   1.0777800601488426e-01   0.0000000000000000e+00
           8.3465143106553319e-01   1.0471633325123153e-01   0.0000000000000000e+00
           8.2669639277606366e-01   1.0170369534475480e-01   0.0000000000000000e+00
           8.1874989394241815e-01   9.8659152046269100e-02   0.0000000000000000e+00
           8.1082521361314908e-01   9.5589480024985407e-02   0.0000000000000000e+00
           8.0291243524621259e-01   9.2477562506154257e-02   0.0000000000000000e+00
           7.9501601582959724e-01   8.9390544379782902e-02   0.0000000000000000e+00
           7.8713591423470530e-01   8.6218414092508924e-02   0.0000000000000000e+00
           7.7926327536719753e-01   8.3082652674223073e-02   0.0000000000000000e+00
           7.7141345413247508e-01   7.9904163323746544e-02   0.0000000000000000e+00
           7.6356901068481819e-01   7.6683348793519776e-02   0.0000000000000000e+00
           7.5574754344824191e-01   7.3559378952035598e-02   0.0000000000000000e+00
           7.4794071009249607e-01   7.0232910613491642e-02   0.0000000000000000e+00
           7.4014602737794621e-01   6.6954364586541498e-02   0.0000000000000000e+00
           7.3236760838059556e-01   6.3633307967642341e-02   0.0000000000000000e+00
           7.2460119293990866e-01   6.0280688573269070e-02   0.0000000000000000e+00
           7.1685625358364513e-01   5.6849608855938531e-02   0.0000000000000000e+00
           7.0912481497916180e-01   5.3299226112309528e-02   0.0000000000000000e+00
           7.0140312192465981e-01   4.9829293199622995e-02   0.0000000000000000e+00
           6.9370510178801092e-01   4.6158764888123434e-02   0.0000000000000000e+00
           6.8602792806999691e-01   4.2524088717921474e-02   0.0000000000000000e+00
           6.7836388922453772e-01   3.8797586133169637e-02   0.0000000000000000e+00
           6.7072178335076416e-01   3.5066982469895418e-02   0.0000000000000000e+00
           6.6309318632075742e-01   3.1291508120498114e-02   0.0000000000000000e+00
           6.5547739533880844e-01   2.7799830769093408e-02   0.0000000000000000e+00
           6.4788052533085139e-01   2.4415535891618653e-02   0.0000000000000000e+00
           6.4031120522707030e-01   2.1137242643728692e-02   0.0000000000000000e+00
           6.3275181226008126e-01   1.7963580322279168e-02   0.0000000000000000e+00
           6.2521102502886350e-01   1.4892646934414970e-02   0.0000000000000000e+00
           6.1768851440583761e-01   1.1947271916774600e-02   0.0000000000000000e+00
           6.1018680816921134e-01   8.8617158086171516e-03   0.0000000000000000e+00
           6.0270479360661422e-01   6.0916955702822869e-03   0.0000000000000000e+00
           5.9523986538913543e-01   3.4203333573827339e-03   0.0000000000000000e+00
           5.8779123408662970e-01   8.4517490685311037e-04   0.0000000000000000e+00
           5.8037263574738407e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.7296951136703345e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.6557284013198694e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.5820630194758614e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.5086745375910373e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.4354221465215014e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.3623817219036729e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.2896046694295906e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.2168511872923218e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.1445545856746699e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.0723090697621742e-01   0.0000000000000000e+00   0.0000000000000000e+00
           5.0002769723112239e-01   0.0000000000000000e+00   0.0000000000000000e+00];
    case 'blue'
        Data=[...
           3.1372549019607843e-02   1.8823529411764706e-01   4.1960784313725491e-01
           3.1372549019607843e-02   1.9215686274509805e-01   4.2745098039215684e-01
           3.1372549019607843e-02   1.9607843137254902e-01   4.3529411764705883e-01
           3.1372549019607843e-02   2.0000000000000001e-01   4.4313725490196076e-01
           3.1372549019607843e-02   2.0392156862745098e-01   4.4705882352941179e-01
           3.1372549019607843e-02   2.0784313725490197e-01   4.5490196078431372e-01
           3.1372549019607843e-02   2.1176470588235294e-01   4.6274509803921571e-01
           3.1372549019607843e-02   2.1568627450980393e-01   4.7058823529411764e-01
           3.1372549019607843e-02   2.1960784313725490e-01   4.7843137254901963e-01
           3.1372549019607843e-02   2.2352941176470589e-01   4.8235294117647060e-01
           3.1372549019607843e-02   2.2745098039215686e-01   4.9019607843137253e-01
           3.1372549019607843e-02   2.3137254901960785e-01   4.9803921568627452e-01
           3.1372549019607843e-02   2.3529411764705882e-01   5.0588235294117645e-01
           3.1372549019607843e-02   2.4313725490196078e-01   5.0980392156862742e-01
           3.1372549019607843e-02   2.4705882352941178e-01   5.1764705882352946e-01
           3.1372549019607843e-02   2.5098039215686274e-01   5.2156862745098043e-01
           3.1372549019607843e-02   2.5490196078431371e-01   5.2941176470588236e-01
           3.1372549019607843e-02   2.5882352941176473e-01   5.3725490196078429e-01
           3.1372549019607843e-02   2.6274509803921570e-01   5.4117647058823526e-01
           3.1372549019607843e-02   2.6666666666666666e-01   5.4901960784313730e-01
           3.1372549019607843e-02   2.7058823529411763e-01   5.5294117647058827e-01
           3.1372549019607843e-02   2.7450980392156865e-01   5.5686274509803924e-01
           3.1372549019607843e-02   2.7843137254901962e-01   5.6470588235294117e-01
           3.1372549019607843e-02   2.8235294117647058e-01   5.6862745098039214e-01
           3.1372549019607843e-02   2.8627450980392155e-01   5.7647058823529407e-01
           3.1372549019607843e-02   2.9019607843137257e-01   5.8039215686274515e-01
           3.1372549019607843e-02   2.9411764705882354e-01   5.8431372549019611e-01
           3.1372549019607843e-02   2.9803921568627451e-01   5.8823529411764708e-01
           3.1372549019607843e-02   3.0196078431372547e-01   5.9607843137254901e-01
           3.1372549019607843e-02   3.0588235294117649e-01   5.9999999999999998e-01
           3.1372549019607843e-02   3.0980392156862746e-01   6.0392156862745094e-01
           3.1372549019607843e-02   3.1372549019607843e-01   6.0784313725490191e-01
           3.1372549019607843e-02   3.1764705882352939e-01   6.1176470588235299e-01
           3.1372549019607843e-02   3.2156862745098042e-01   6.1568627450980395e-01
           3.1372549019607843e-02   3.2549019607843138e-01   6.1960784313725492e-01
           3.1372549019607843e-02   3.2941176470588235e-01   6.2352941176470589e-01
           3.5294117647058823e-02   3.3333333333333331e-01   6.2745098039215685e-01
           3.5294117647058823e-02   3.3725490196078434e-01   6.3137254901960782e-01
           3.5294117647058823e-02   3.4117647058823530e-01   6.3529411764705879e-01
           3.9215686274509803e-02   3.4509803921568627e-01   6.3921568627450975e-01
           4.3137254901960784e-02   3.4901960784313724e-01   6.4313725490196083e-01
           4.3137254901960784e-02   3.5294117647058826e-01   6.4313725490196083e-01
           4.7058823529411764e-02   3.5686274509803922e-01   6.4705882352941180e-01
           5.0980392156862744e-02   3.6078431372549019e-01   6.5098039215686276e-01
           5.0980392156862744e-02   3.6470588235294116e-01   6.5490196078431373e-01
           5.4901960784313725e-02   3.6862745098039218e-01   6.5882352941176470e-01
           5.8823529411764705e-02   3.7254901960784315e-01   6.6274509803921566e-01
           6.2745098039215685e-02   3.7647058823529411e-01   6.6274509803921566e-01
           6.6666666666666666e-02   3.8039215686274508e-01   6.6666666666666663e-01
           7.0588235294117646e-02   3.8431372549019610e-01   6.7058823529411760e-01
           7.4509803921568626e-02   3.8823529411764707e-01   6.7450980392156867e-01
           7.8431372549019607e-02   3.9215686274509803e-01   6.7450980392156867e-01
           8.2352941176470587e-02   3.9607843137254900e-01   6.7843137254901964e-01
           8.6274509803921567e-02   4.0000000000000002e-01   6.8235294117647061e-01
           9.0196078431372548e-02   4.0392156862745099e-01   6.8235294117647061e-01
           9.4117647058823528e-02   4.0784313725490196e-01   6.8627450980392157e-01
           9.8039215686274508e-02   4.1176470588235292e-01   6.9019607843137254e-01
           1.0196078431372549e-01   4.1568627450980394e-01   6.9019607843137254e-01
           1.0588235294117647e-01   4.1960784313725491e-01   6.9411764705882351e-01
           1.0980392156862745e-01   4.2352941176470588e-01   6.9803921568627447e-01
           1.1372549019607843e-01   4.2745098039215684e-01   6.9803921568627447e-01
           1.1764705882352941e-01   4.3137254901960786e-01   7.0196078431372544e-01
           1.2156862745098039e-01   4.3529411764705883e-01   7.0588235294117652e-01
           1.2549019607843137e-01   4.3921568627450980e-01   7.0588235294117652e-01
           1.2941176470588237e-01   4.4313725490196076e-01   7.0980392156862748e-01
           1.3333333333333333e-01   4.4705882352941179e-01   7.1372549019607845e-01
           1.3725490196078433e-01   4.5098039215686275e-01   7.1372549019607845e-01
           1.4117647058823529e-01   4.5490196078431372e-01   7.1764705882352942e-01
           1.4509803921568629e-01   4.5882352941176469e-01   7.1764705882352942e-01
           1.4901960784313725e-01   4.6274509803921571e-01   7.2156862745098038e-01
           1.5294117647058825e-01   4.6666666666666667e-01   7.2549019607843135e-01
           1.5294117647058825e-01   4.7058823529411764e-01   7.2549019607843135e-01
           1.5686274509803921e-01   4.7450980392156861e-01   7.2941176470588232e-01
           1.6078431372549021e-01   4.7843137254901963e-01   7.2941176470588232e-01
           1.6470588235294117e-01   4.8235294117647060e-01   7.3333333333333328e-01
           1.6862745098039217e-01   4.8627450980392156e-01   7.3333333333333328e-01
           1.7254901960784313e-01   4.9019607843137253e-01   7.3725490196078436e-01
           1.7647058823529413e-01   4.9803921568627452e-01   7.3725490196078436e-01
           1.8039215686274510e-01   5.0196078431372548e-01   7.4117647058823533e-01
           1.8431372549019609e-01   5.0588235294117645e-01   7.4117647058823533e-01
           1.8823529411764706e-01   5.0980392156862742e-01   7.4509803921568629e-01
           1.9215686274509805e-01   5.1372549019607838e-01   7.4509803921568629e-01
           2.0000000000000001e-01   5.1764705882352946e-01   7.4901960784313726e-01
           2.0392156862745098e-01   5.2156862745098043e-01   7.4901960784313726e-01
           2.0784313725490197e-01   5.2549019607843139e-01   7.5294117647058822e-01
           2.1176470588235294e-01   5.2941176470588236e-01   7.5294117647058822e-01
           2.1568627450980393e-01   5.3333333333333333e-01   7.5686274509803919e-01
           2.1960784313725490e-01   5.3725490196078429e-01   7.5686274509803919e-01
           2.2352941176470589e-01   5.4117647058823526e-01   7.6078431372549016e-01
           2.2745098039215686e-01   5.4509803921568623e-01   7.6078431372549016e-01
           2.3137254901960785e-01   5.4901960784313730e-01   7.6470588235294112e-01
           2.3529411764705882e-01   5.5294117647058827e-01   7.6470588235294112e-01
           2.3921568627450981e-01   5.5686274509803924e-01   7.6862745098039220e-01
           2.4705882352941178e-01   5.6078431372549020e-01   7.6862745098039220e-01
           2.5098039215686274e-01   5.6470588235294117e-01   7.7254901960784317e-01
           2.5490196078431371e-01   5.6862745098039214e-01   7.7254901960784317e-01
           2.5882352941176473e-01   5.7254901960784310e-01   7.7647058823529413e-01
           2.6274509803921570e-01   5.7647058823529407e-01   7.8039215686274510e-01
           2.6666666666666666e-01   5.8039215686274515e-01   7.8039215686274510e-01
           2.7450980392156865e-01   5.8431372549019611e-01   7.8431372549019607e-01
           2.7843137254901962e-01   5.8823529411764708e-01   7.8431372549019607e-01
           2.8235294117647058e-01   5.9215686274509804e-01   7.8823529411764703e-01
           2.8627450980392155e-01   5.9607843137254901e-01   7.8823529411764703e-01
           2.9019607843137257e-01   5.9999999999999998e-01   7.9215686274509800e-01
           2.9803921568627451e-01   6.0392156862745094e-01   7.9215686274509800e-01
           3.0196078431372547e-01   6.0392156862745094e-01   7.9607843137254897e-01
           3.0588235294117649e-01   6.0784313725490191e-01   7.9607843137254897e-01
           3.0980392156862746e-01   6.1176470588235299e-01   8.0000000000000004e-01
           3.1764705882352939e-01   6.1568627450980395e-01   8.0392156862745101e-01
           3.2156862745098042e-01   6.1960784313725492e-01   8.0392156862745101e-01
           3.2549019607843138e-01   6.2352941176470589e-01   8.0784313725490198e-01
           3.3333333333333331e-01   6.2745098039215685e-01   8.0784313725490198e-01
           3.3725490196078434e-01   6.3137254901960782e-01   8.1176470588235294e-01
           3.4117647058823530e-01   6.3529411764705879e-01   8.1176470588235294e-01
           3.4901960784313724e-01   6.3921568627450975e-01   8.1568627450980391e-01
           3.5294117647058826e-01   6.3921568627450975e-01   8.1568627450980391e-01
           3.5686274509803922e-01   6.4313725490196083e-01   8.1960784313725488e-01
           3.6470588235294116e-01   6.4705882352941180e-01   8.1960784313725488e-01
           3.6862745098039218e-01   6.5098039215686276e-01   8.2352941176470584e-01
           3.7647058823529411e-01   6.5490196078431373e-01   8.2352941176470584e-01
           3.8039215686274508e-01   6.5882352941176470e-01   8.2745098039215681e-01
           3.8431372549019610e-01   6.6274509803921566e-01   8.2745098039215681e-01
           3.9215686274509803e-01   6.6666666666666663e-01   8.3137254901960789e-01
           3.9607843137254900e-01   6.6666666666666663e-01   8.3137254901960789e-01
           4.0392156862745099e-01   6.7058823529411760e-01   8.3529411764705885e-01
           4.0784313725490196e-01   6.7450980392156867e-01   8.3529411764705885e-01
           4.1568627450980394e-01   6.7843137254901964e-01   8.3921568627450982e-01
           4.1960784313725491e-01   6.8235294117647061e-01   8.3921568627450982e-01
           4.2352941176470588e-01   6.8627450980392157e-01   8.3921568627450982e-01
           4.3137254901960786e-01   6.9019607843137254e-01   8.4313725490196079e-01
           4.3529411764705883e-01   6.9411764705882351e-01   8.4313725490196079e-01
           4.4313725490196076e-01   6.9803921568627447e-01   8.4705882352941175e-01
           4.5098039215686275e-01   7.0196078431372544e-01   8.4705882352941175e-01
           4.5490196078431372e-01   7.0588235294117652e-01   8.4705882352941175e-01
           4.6274509803921571e-01   7.0588235294117652e-01   8.5098039215686272e-01
           4.6666666666666667e-01   7.0980392156862748e-01   8.5098039215686272e-01
           4.7450980392156861e-01   7.1372549019607845e-01   8.5098039215686272e-01
           4.8235294117647060e-01   7.1764705882352942e-01   8.5490196078431369e-01
           4.8627450980392156e-01   7.2156862745098038e-01   8.5490196078431369e-01
           4.9411764705882355e-01   7.2549019607843135e-01   8.5490196078431369e-01
           5.0196078431372548e-01   7.2941176470588232e-01   8.5882352941176465e-01
           5.0588235294117645e-01   7.3333333333333328e-01   8.5882352941176465e-01
           5.1372549019607838e-01   7.3725490196078436e-01   8.5882352941176465e-01
           5.2156862745098043e-01   7.4117647058823533e-01   8.6274509803921573e-01
           5.2549019607843139e-01   7.4509803921568629e-01   8.6274509803921573e-01
           5.3333333333333333e-01   7.4901960784313726e-01   8.6274509803921573e-01
           5.4117647058823526e-01   7.5294117647058822e-01   8.6666666666666670e-01
           5.4509803921568623e-01   7.5686274509803919e-01   8.6666666666666670e-01
           5.5294117647058827e-01   7.5686274509803919e-01   8.6666666666666670e-01
           5.6078431372549020e-01   7.6078431372549016e-01   8.7058823529411766e-01
           5.6470588235294117e-01   7.6470588235294112e-01   8.7058823529411766e-01
           5.7254901960784310e-01   7.6862745098039220e-01   8.7058823529411766e-01
           5.7647058823529407e-01   7.7254901960784317e-01   8.7450980392156863e-01
           5.8431372549019611e-01   7.7647058823529413e-01   8.7450980392156863e-01
           5.9215686274509804e-01   7.7647058823529413e-01   8.7450980392156863e-01
           5.9607843137254901e-01   7.8039215686274510e-01   8.7843137254901960e-01
           6.0392156862745094e-01   7.8431372549019607e-01   8.7843137254901960e-01
           6.0784313725490191e-01   7.8823529411764703e-01   8.7843137254901960e-01
           6.1568627450980395e-01   7.8823529411764703e-01   8.8235294117647056e-01
           6.1960784313725492e-01   7.9215686274509800e-01   8.8235294117647056e-01
           6.2352941176470589e-01   7.9607843137254897e-01   8.8235294117647056e-01
           6.3137254901960782e-01   7.9607843137254897e-01   8.8627450980392153e-01
           6.3529411764705879e-01   8.0000000000000004e-01   8.8627450980392153e-01
           6.4313725490196083e-01   8.0392156862745101e-01   8.9019607843137250e-01
           6.4705882352941180e-01   8.0392156862745101e-01   8.9019607843137250e-01
           6.5098039215686276e-01   8.0784313725490198e-01   8.9411764705882357e-01
           6.5882352941176470e-01   8.0784313725490198e-01   8.9411764705882357e-01
           6.6274509803921566e-01   8.1176470588235294e-01   8.9411764705882357e-01
           6.7058823529411760e-01   8.1176470588235294e-01   8.9803921568627454e-01
           6.7450980392156867e-01   8.1568627450980391e-01   8.9803921568627454e-01
           6.7843137254901964e-01   8.1568627450980391e-01   9.0196078431372551e-01
           6.8627450980392157e-01   8.1960784313725488e-01   9.0196078431372551e-01
           6.9019607843137254e-01   8.2352941176470584e-01   9.0588235294117647e-01
           6.9411764705882351e-01   8.2352941176470584e-01   9.0588235294117647e-01
           7.0196078431372544e-01   8.2745098039215681e-01   9.0980392156862744e-01
           7.0588235294117652e-01   8.2745098039215681e-01   9.0980392156862744e-01
           7.0980392156862748e-01   8.3137254901960789e-01   9.1372549019607840e-01
           7.1372549019607845e-01   8.3137254901960789e-01   9.1372549019607840e-01
           7.2156862745098038e-01   8.3529411764705885e-01   9.1764705882352937e-01
           7.2549019607843135e-01   8.3529411764705885e-01   9.1764705882352937e-01
           7.2941176470588232e-01   8.3921568627450982e-01   9.2156862745098034e-01
           7.3333333333333328e-01   8.3921568627450982e-01   9.2156862745098034e-01
           7.3725490196078436e-01   8.4313725490196079e-01   9.2549019607843142e-01
           7.4509803921568629e-01   8.4313725490196079e-01   9.2549019607843142e-01
           7.4901960784313726e-01   8.4705882352941175e-01   9.2549019607843142e-01
           7.5294117647058822e-01   8.4705882352941175e-01   9.2941176470588238e-01
           7.5686274509803919e-01   8.4705882352941175e-01   9.2941176470588238e-01
           7.6078431372549016e-01   8.5098039215686272e-01   9.3333333333333335e-01
           7.6470588235294112e-01   8.5098039215686272e-01   9.3333333333333335e-01
           7.6862745098039220e-01   8.5490196078431369e-01   9.3333333333333335e-01
           7.7254901960784317e-01   8.5490196078431369e-01   9.3725490196078431e-01
           7.7647058823529413e-01   8.5882352941176465e-01   9.3725490196078431e-01
           7.8039215686274510e-01   8.6274509803921573e-01   9.3725490196078431e-01
           7.8431372549019607e-01   8.6274509803921573e-01   9.4117647058823528e-01
           7.8823529411764703e-01   8.6666666666666670e-01   9.4117647058823528e-01
           7.9215686274509800e-01   8.6666666666666670e-01   9.4117647058823528e-01
           7.9215686274509800e-01   8.7058823529411766e-01   9.4117647058823528e-01
           7.9607843137254897e-01   8.7058823529411766e-01   9.4509803921568625e-01
           8.0000000000000004e-01   8.7450980392156863e-01   9.4509803921568625e-01
           8.0392156862745101e-01   8.7450980392156863e-01   9.4509803921568625e-01
           8.0784313725490198e-01   8.7843137254901960e-01   9.4901960784313721e-01
           8.0784313725490198e-01   8.7843137254901960e-01   9.4901960784313721e-01
           8.1176470588235294e-01   8.8235294117647056e-01   9.4901960784313721e-01
           8.1568627450980391e-01   8.8235294117647056e-01   9.4901960784313721e-01
           8.1960784313725488e-01   8.8627450980392153e-01   9.5294117647058818e-01
           8.1960784313725488e-01   8.8627450980392153e-01   9.5294117647058818e-01
           8.2352941176470584e-01   8.9019607843137250e-01   9.5294117647058818e-01
           8.2745098039215681e-01   8.9019607843137250e-01   9.5294117647058818e-01
           8.2745098039215681e-01   8.9411764705882357e-01   9.5686274509803926e-01
           8.3137254901960789e-01   8.9411764705882357e-01   9.5686274509803926e-01
           8.3529411764705885e-01   8.9803921568627454e-01   9.5686274509803926e-01
           8.3529411764705885e-01   8.9803921568627454e-01   9.5686274509803926e-01
           8.3921568627450982e-01   9.0196078431372551e-01   9.5686274509803926e-01
           8.4313725490196079e-01   9.0196078431372551e-01   9.6078431372549022e-01
           8.4705882352941175e-01   9.0588235294117647e-01   9.6078431372549022e-01
           8.4705882352941175e-01   9.0588235294117647e-01   9.6078431372549022e-01
           8.5098039215686272e-01   9.0980392156862744e-01   9.6078431372549022e-01
           8.5490196078431369e-01   9.0980392156862744e-01   9.6470588235294119e-01
           8.5490196078431369e-01   9.1372549019607840e-01   9.6470588235294119e-01
           8.5882352941176465e-01   9.1372549019607840e-01   9.6470588235294119e-01
           8.6274509803921573e-01   9.1764705882352937e-01   9.6470588235294119e-01
           8.6274509803921573e-01   9.1764705882352937e-01   9.6862745098039216e-01
           8.6666666666666670e-01   9.2156862745098034e-01   9.6862745098039216e-01
           8.7058823529411766e-01   9.2156862745098034e-01   9.6862745098039216e-01
           8.7450980392156863e-01   9.2549019607843142e-01   9.6862745098039216e-01
           8.7843137254901960e-01   9.2549019607843142e-01   9.7254901960784312e-01
           8.7843137254901960e-01   9.2941176470588238e-01   9.7254901960784312e-01
           8.8235294117647056e-01   9.2941176470588238e-01   9.7254901960784312e-01
           8.8627450980392153e-01   9.3333333333333335e-01   9.7254901960784312e-01
           8.9019607843137250e-01   9.3333333333333335e-01   9.7647058823529409e-01
           8.9019607843137250e-01   9.3725490196078431e-01   9.7647058823529409e-01
           8.9411764705882357e-01   9.3725490196078431e-01   9.7647058823529409e-01
           8.9803921568627454e-01   9.4117647058823528e-01   9.7647058823529409e-01
           9.0196078431372551e-01   9.4117647058823528e-01   9.8039215686274506e-01
           9.0196078431372551e-01   9.4509803921568625e-01   9.8039215686274506e-01
           9.0588235294117647e-01   9.4509803921568625e-01   9.8039215686274506e-01
           9.0980392156862744e-01   9.4901960784313721e-01   9.8039215686274506e-01
           9.1372549019607840e-01   9.4901960784313721e-01   9.8431372549019602e-01
           9.1764705882352937e-01   9.5294117647058818e-01   9.8431372549019602e-01
           9.1764705882352937e-01   9.5294117647058818e-01   9.8431372549019602e-01
           9.2156862745098034e-01   9.5686274509803926e-01   9.8431372549019602e-01
           9.2549019607843142e-01   9.5686274509803926e-01   9.8823529411764710e-01
           9.2941176470588238e-01   9.6078431372549022e-01   9.8823529411764710e-01
           9.3333333333333335e-01   9.6078431372549022e-01   9.8823529411764710e-01
           9.3333333333333335e-01   9.6470588235294119e-01   9.8823529411764710e-01
           9.3725490196078431e-01   9.6470588235294119e-01   9.9215686274509807e-01
           9.4117647058823528e-01   9.6862745098039216e-01   9.9215686274509807e-01
           9.4509803921568625e-01   9.6862745098039216e-01   9.9215686274509807e-01
           9.4509803921568625e-01   9.7254901960784312e-01   9.9215686274509807e-01
           9.4901960784313721e-01   9.7254901960784312e-01   9.9607843137254903e-01
           9.5294117647058818e-01   9.7647058823529409e-01   9.9607843137254903e-01
           9.5686274509803926e-01   9.7647058823529409e-01   9.9607843137254903e-01
           9.6078431372549022e-01   9.8039215686274506e-01   9.9607843137254903e-01
           9.6078431372549022e-01   9.8039215686274506e-01   1.0000000000000000e+00
           9.6470588235294119e-01   9.8431372549019602e-01   1.0000000000000000e+00
           9.6862745098039216e-01   9.8431372549019602e-01   1.0000000000000000e+00];
    case 'green'
        Data=[...
           0.0000000000000000e+00   2.6666666666666666e-01   1.0588235294117647e-01
           0.0000000000000000e+00   2.7058823529411763e-01   1.0588235294117647e-01
           0.0000000000000000e+00   2.7843137254901962e-01   1.0980392156862745e-01
           0.0000000000000000e+00   2.8235294117647058e-01   1.0980392156862745e-01
           0.0000000000000000e+00   2.9019607843137257e-01   1.1372549019607843e-01
           0.0000000000000000e+00   2.9411764705882354e-01   1.1372549019607843e-01
           0.0000000000000000e+00   3.0196078431372547e-01   1.1764705882352941e-01
           0.0000000000000000e+00   3.0588235294117649e-01   1.1764705882352941e-01
           0.0000000000000000e+00   3.0980392156862746e-01   1.2156862745098039e-01
           0.0000000000000000e+00   3.1764705882352939e-01   1.2156862745098039e-01
           0.0000000000000000e+00   3.2156862745098042e-01   1.2549019607843137e-01
           0.0000000000000000e+00   3.2549019607843138e-01   1.2549019607843137e-01
           0.0000000000000000e+00   3.3333333333333331e-01   1.2941176470588237e-01
           0.0000000000000000e+00   3.3725490196078434e-01   1.2941176470588237e-01
           0.0000000000000000e+00   3.4117647058823530e-01   1.3333333333333333e-01
           0.0000000000000000e+00   3.4901960784313724e-01   1.3333333333333333e-01
           0.0000000000000000e+00   3.5294117647058826e-01   1.3725490196078433e-01
           0.0000000000000000e+00   3.5686274509803922e-01   1.3725490196078433e-01
           0.0000000000000000e+00   3.6470588235294116e-01   1.4117647058823529e-01
           0.0000000000000000e+00   3.6862745098039218e-01   1.4117647058823529e-01
           0.0000000000000000e+00   3.7254901960784315e-01   1.4509803921568629e-01
           0.0000000000000000e+00   3.7647058823529411e-01   1.4509803921568629e-01
           0.0000000000000000e+00   3.8039215686274508e-01   1.4901960784313725e-01
           0.0000000000000000e+00   3.8823529411764707e-01   1.5294117647058825e-01
           0.0000000000000000e+00   3.9215686274509803e-01   1.5294117647058825e-01
           0.0000000000000000e+00   3.9607843137254900e-01   1.5686274509803921e-01
           0.0000000000000000e+00   4.0000000000000002e-01   1.5686274509803921e-01
           0.0000000000000000e+00   4.0392156862745099e-01   1.6078431372549021e-01
           0.0000000000000000e+00   4.1176470588235292e-01   1.6470588235294117e-01
           0.0000000000000000e+00   4.1568627450980394e-01   1.6470588235294117e-01
           0.0000000000000000e+00   4.1960784313725491e-01   1.6862745098039217e-01
           0.0000000000000000e+00   4.2352941176470588e-01   1.6862745098039217e-01
           0.0000000000000000e+00   4.2745098039215684e-01   1.7254901960784313e-01
           0.0000000000000000e+00   4.3137254901960786e-01   1.7647058823529413e-01
           0.0000000000000000e+00   4.3529411764705883e-01   1.7647058823529413e-01
           3.9215686274509803e-03   4.3921568627450980e-01   1.8039215686274510e-01
           3.9215686274509803e-03   4.4313725490196076e-01   1.8431372549019609e-01
           7.8431372549019607e-03   4.4705882352941179e-01   1.8431372549019609e-01
           7.8431372549019607e-03   4.5098039215686275e-01   1.8823529411764706e-01
           1.1764705882352941e-02   4.5490196078431372e-01   1.9215686274509805e-01
           1.5686274509803921e-02   4.5882352941176469e-01   1.9607843137254902e-01
           1.9607843137254902e-02   4.6274509803921571e-01   1.9607843137254902e-01
           2.3529411764705882e-02   4.6666666666666667e-01   2.0000000000000001e-01
           2.7450980392156862e-02   4.7058823529411764e-01   2.0392156862745098e-01
           3.1372549019607843e-02   4.7450980392156861e-01   2.0784313725490197e-01
           3.5294117647058823e-02   4.7843137254901963e-01   2.1176470588235294e-01
           4.3137254901960784e-02   4.8235294117647060e-01   2.1176470588235294e-01
           4.7058823529411764e-02   4.8627450980392156e-01   2.1568627450980393e-01
           5.0980392156862744e-02   4.8627450980392156e-01   2.1960784313725490e-01
           5.8823529411764705e-02   4.9019607843137253e-01   2.2352941176470589e-01
           6.2745098039215685e-02   4.9411764705882355e-01   2.2745098039215686e-01
           7.0588235294117646e-02   4.9803921568627452e-01   2.2745098039215686e-01
           7.4509803921568626e-02   5.0196078431372548e-01   2.3137254901960785e-01
           8.2352941176470587e-02   5.0588235294117645e-01   2.3529411764705882e-01
           8.6274509803921567e-02   5.0980392156862742e-01   2.3921568627450981e-01
           9.4117647058823528e-02   5.1372549019607838e-01   2.4313725490196078e-01
           9.8039215686274508e-02   5.1764705882352946e-01   2.4705882352941178e-01
           1.0196078431372549e-01   5.1764705882352946e-01   2.4705882352941178e-01
           1.0980392156862745e-01   5.2156862745098043e-01   2.5098039215686274e-01
           1.1372549019607843e-01   5.2549019607843139e-01   2.5490196078431371e-01
           1.1764705882352941e-01   5.2941176470588236e-01   2.5882352941176473e-01
           1.2549019607843137e-01   5.3333333333333333e-01   2.6274509803921570e-01
           1.2941176470588237e-01   5.3725490196078429e-01   2.6274509803921570e-01
           1.3333333333333333e-01   5.4117647058823526e-01   2.6666666666666666e-01
           1.3725490196078433e-01   5.4509803921568623e-01   2.7058823529411763e-01
           1.4117647058823529e-01   5.4901960784313730e-01   2.7450980392156865e-01
           1.4509803921568629e-01   5.5294117647058827e-01   2.7843137254901962e-01
           1.4901960784313725e-01   5.5686274509803924e-01   2.7843137254901962e-01
           1.5294117647058825e-01   5.6078431372549020e-01   2.8235294117647058e-01
           1.5686274509803921e-01   5.6470588235294117e-01   2.8627450980392155e-01
           1.6078431372549021e-01   5.6862745098039214e-01   2.9019607843137257e-01
           1.6470588235294117e-01   5.7254901960784310e-01   2.9019607843137257e-01
           1.6470588235294117e-01   5.7647058823529407e-01   2.9411764705882354e-01
           1.6862745098039217e-01   5.8039215686274515e-01   2.9803921568627451e-01
           1.7254901960784313e-01   5.8431372549019611e-01   3.0196078431372547e-01
           1.7647058823529413e-01   5.8823529411764708e-01   3.0196078431372547e-01
           1.8039215686274510e-01   5.9215686274509804e-01   3.0588235294117649e-01
           1.8431372549019609e-01   5.9607843137254901e-01   3.0980392156862746e-01
           1.8823529411764706e-01   5.9999999999999998e-01   3.0980392156862746e-01
           1.8823529411764706e-01   6.0392156862745094e-01   3.1372549019607843e-01
           1.9215686274509805e-01   6.0784313725490191e-01   3.1764705882352939e-01
           1.9607843137254902e-01   6.1176470588235299e-01   3.2156862745098042e-01
           2.0000000000000001e-01   6.1568627450980395e-01   3.2156862745098042e-01
           2.0392156862745098e-01   6.1960784313725492e-01   3.2549019607843138e-01
           2.0784313725490197e-01   6.2352941176470589e-01   3.2941176470588235e-01
           2.1176470588235294e-01   6.2745098039215685e-01   3.3333333333333331e-01
           2.1568627450980393e-01   6.3137254901960782e-01   3.3333333333333331e-01
           2.1568627450980393e-01   6.3529411764705879e-01   3.3725490196078434e-01
           2.1960784313725490e-01   6.3921568627450975e-01   3.4117647058823530e-01
           2.2352941176470589e-01   6.4313725490196083e-01   3.4509803921568627e-01
           2.2745098039215686e-01   6.4705882352941180e-01   3.4509803921568627e-01
           2.3137254901960785e-01   6.5098039215686276e-01   3.4901960784313724e-01
           2.3529411764705882e-01   6.5490196078431373e-01   3.5294117647058826e-01
           2.4313725490196078e-01   6.5882352941176470e-01   3.5686274509803922e-01
           2.4705882352941178e-01   6.6274509803921566e-01   3.5686274509803922e-01
           2.5098039215686274e-01   6.6666666666666663e-01   3.6078431372549019e-01
           2.5490196078431371e-01   6.7058823529411760e-01   3.6470588235294116e-01
           2.5882352941176473e-01   6.7450980392156867e-01   3.6862745098039218e-01
           2.6666666666666666e-01   6.7843137254901964e-01   3.7254901960784315e-01
           2.7058823529411763e-01   6.8235294117647061e-01   3.7254901960784315e-01
           2.7450980392156865e-01   6.8627450980392157e-01   3.7647058823529411e-01
           2.8235294117647058e-01   6.8627450980392157e-01   3.8039215686274508e-01
           2.8627450980392155e-01   6.9019607843137254e-01   3.8431372549019610e-01
           2.9411764705882354e-01   6.9411764705882351e-01   3.8431372549019610e-01
           2.9803921568627451e-01   6.9803921568627447e-01   3.8823529411764707e-01
           3.0588235294117649e-01   7.0196078431372544e-01   3.9215686274509803e-01
           3.1372549019607843e-01   7.0588235294117652e-01   3.9607843137254900e-01
           3.1764705882352939e-01   7.0588235294117652e-01   3.9607843137254900e-01
           3.2549019607843138e-01   7.0980392156862748e-01   4.0000000000000002e-01
           3.3333333333333331e-01   7.1372549019607845e-01   4.0392156862745099e-01
           3.4117647058823530e-01   7.1764705882352942e-01   4.0784313725490196e-01
           3.4509803921568627e-01   7.2156862745098038e-01   4.0784313725490196e-01
           3.5294117647058826e-01   7.2549019607843135e-01   4.1176470588235292e-01
           3.6078431372549019e-01   7.2549019607843135e-01   4.1568627450980394e-01
           3.6862745098039218e-01   7.2941176470588232e-01   4.1960784313725491e-01
           3.7647058823529411e-01   7.3333333333333328e-01   4.2352941176470588e-01
           3.8039215686274508e-01   7.3725490196078436e-01   4.2352941176470588e-01
           3.8823529411764707e-01   7.4117647058823533e-01   4.2745098039215684e-01
           3.9607843137254900e-01   7.4117647058823533e-01   4.3137254901960786e-01
           4.0392156862745099e-01   7.4509803921568629e-01   4.3529411764705883e-01
           4.0784313725490196e-01   7.4901960784313726e-01   4.3921568627450980e-01
           4.1568627450980394e-01   7.5294117647058822e-01   4.4313725490196076e-01
           4.2352941176470588e-01   7.5294117647058822e-01   4.4313725490196076e-01
           4.3137254901960786e-01   7.5686274509803919e-01   4.4705882352941179e-01
           4.3529411764705883e-01   7.6078431372549016e-01   4.5098039215686275e-01
           4.4313725490196076e-01   7.6470588235294112e-01   4.5490196078431372e-01
           4.4705882352941179e-01   7.6470588235294112e-01   4.5882352941176469e-01
           4.5490196078431372e-01   7.6862745098039220e-01   4.6274509803921571e-01
           4.6274509803921571e-01   7.7254901960784317e-01   4.6666666666666667e-01
           4.6666666666666667e-01   7.7254901960784317e-01   4.7058823529411764e-01
           4.7450980392156861e-01   7.7647058823529413e-01   4.7450980392156861e-01
           4.7843137254901963e-01   7.8039215686274510e-01   4.7843137254901963e-01
           4.8627450980392156e-01   7.8431372549019607e-01   4.8235294117647060e-01
           4.9019607843137253e-01   7.8431372549019607e-01   4.8627450980392156e-01
           4.9411764705882355e-01   7.8823529411764703e-01   4.9019607843137253e-01
           5.0196078431372548e-01   7.9215686274509800e-01   4.9411764705882355e-01
           5.0588235294117645e-01   7.9215686274509800e-01   4.9803921568627452e-01
           5.1372549019607838e-01   7.9607843137254897e-01   5.0588235294117645e-01
           5.1764705882352946e-01   8.0000000000000004e-01   5.0980392156862742e-01
           5.2549019607843139e-01   8.0000000000000004e-01   5.1372549019607838e-01
           5.2941176470588236e-01   8.0392156862745101e-01   5.1764705882352946e-01
           5.3725490196078429e-01   8.0784313725490198e-01   5.2156862745098043e-01
           5.4117647058823526e-01   8.0784313725490198e-01   5.2549019607843139e-01
           5.4509803921568623e-01   8.1176470588235294e-01   5.3333333333333333e-01
           5.5294117647058827e-01   8.1568627450980391e-01   5.3725490196078429e-01
           5.5686274509803924e-01   8.1568627450980391e-01   5.4117647058823526e-01
           5.6470588235294117e-01   8.1960784313725488e-01   5.4509803921568623e-01
           5.6862745098039214e-01   8.2352941176470584e-01   5.5294117647058827e-01
           5.7254901960784310e-01   8.2352941176470584e-01   5.5686274509803924e-01
           5.8039215686274515e-01   8.2745098039215681e-01   5.6078431372549020e-01
           5.8431372549019611e-01   8.3137254901960789e-01   5.6470588235294117e-01
           5.8823529411764708e-01   8.3137254901960789e-01   5.6862745098039214e-01
           5.9607843137254901e-01   8.3529411764705885e-01   5.7647058823529407e-01
           5.9999999999999998e-01   8.3529411764705885e-01   5.8039215686274515e-01
           6.0392156862745094e-01   8.3921568627450982e-01   5.8431372549019611e-01
           6.1176470588235299e-01   8.4313725490196079e-01   5.8823529411764708e-01
           6.1568627450980395e-01   8.4313725490196079e-01   5.9215686274509804e-01
           6.1960784313725492e-01   8.4705882352941175e-01   5.9999999999999998e-01
           6.2745098039215685e-01   8.4705882352941175e-01   6.0392156862745094e-01
           6.3137254901960782e-01   8.5098039215686272e-01   6.0784313725490191e-01
           6.3529411764705879e-01   8.5490196078431369e-01   6.1176470588235299e-01
           6.4313725490196083e-01   8.5490196078431369e-01   6.1568627450980395e-01
           6.4705882352941180e-01   8.5882352941176465e-01   6.1960784313725492e-01
           6.5098039215686276e-01   8.5882352941176465e-01   6.2745098039215685e-01
           6.5490196078431373e-01   8.6274509803921573e-01   6.3137254901960782e-01
           6.6274509803921566e-01   8.6274509803921573e-01   6.3529411764705879e-01
           6.6666666666666663e-01   8.6666666666666670e-01   6.3921568627450975e-01
           6.7058823529411760e-01   8.6666666666666670e-01   6.4313725490196083e-01
           6.7450980392156867e-01   8.7058823529411766e-01   6.5098039215686276e-01
           6.8235294117647061e-01   8.7058823529411766e-01   6.5490196078431373e-01
           6.8627450980392157e-01   8.7450980392156863e-01   6.5882352941176470e-01
           6.9019607843137254e-01   8.7843137254901960e-01   6.6274509803921566e-01
           6.9411764705882351e-01   8.7843137254901960e-01   6.6666666666666663e-01
           7.0196078431372544e-01   8.8235294117647056e-01   6.7058823529411760e-01
           7.0588235294117652e-01   8.8235294117647056e-01   6.7843137254901964e-01
           7.0980392156862748e-01   8.8627450980392153e-01   6.8235294117647061e-01
           7.1372549019607845e-01   8.8627450980392153e-01   6.8627450980392157e-01
           7.1764705882352942e-01   8.9019607843137250e-01   6.9019607843137254e-01
           7.2549019607843135e-01   8.9019607843137250e-01   6.9411764705882351e-01
           7.2941176470588232e-01   8.9411764705882357e-01   7.0196078431372544e-01
           7.3333333333333328e-01   8.9411764705882357e-01   7.0588235294117652e-01
           7.3725490196078436e-01   8.9411764705882357e-01   7.0980392156862748e-01
           7.4117647058823533e-01   8.9803921568627454e-01   7.1372549019607845e-01
           7.4509803921568629e-01   8.9803921568627454e-01   7.1764705882352942e-01
           7.4901960784313726e-01   9.0196078431372551e-01   7.2156862745098038e-01
           7.5686274509803919e-01   9.0196078431372551e-01   7.2549019607843135e-01
           7.6078431372549016e-01   9.0588235294117647e-01   7.3333333333333328e-01
           7.6470588235294112e-01   9.0588235294117647e-01   7.3725490196078436e-01
           7.6862745098039220e-01   9.0980392156862744e-01   7.4117647058823533e-01
           7.7254901960784317e-01   9.0980392156862744e-01   7.4509803921568629e-01
           7.7647058823529413e-01   9.1372549019607840e-01   7.4901960784313726e-01
           7.8039215686274510e-01   9.1372549019607840e-01   7.5294117647058822e-01
           7.8431372549019607e-01   9.1372549019607840e-01   7.5686274509803919e-01
           7.8823529411764703e-01   9.1764705882352937e-01   7.6078431372549016e-01
           7.9215686274509800e-01   9.1764705882352937e-01   7.6470588235294112e-01
           7.9607843137254897e-01   9.2156862745098034e-01   7.6862745098039220e-01
           8.0000000000000004e-01   9.2156862745098034e-01   7.7254901960784317e-01
           8.0392156862745101e-01   9.2549019607843142e-01   7.7647058823529413e-01
           8.0784313725490198e-01   9.2549019607843142e-01   7.8431372549019607e-01
           8.1176470588235294e-01   9.2549019607843142e-01   7.8823529411764703e-01
           8.1568627450980391e-01   9.2941176470588238e-01   7.9215686274509800e-01
           8.1960784313725488e-01   9.2941176470588238e-01   7.9607843137254897e-01
           8.2352941176470584e-01   9.3333333333333335e-01   8.0000000000000004e-01
           8.2745098039215681e-01   9.3333333333333335e-01   8.0392156862745101e-01
           8.3137254901960789e-01   9.3333333333333335e-01   8.0784313725490198e-01
           8.3529411764705885e-01   9.3725490196078431e-01   8.1176470588235294e-01
           8.3921568627450982e-01   9.3725490196078431e-01   8.1568627450980391e-01
           8.4313725490196079e-01   9.4117647058823528e-01   8.1960784313725488e-01
           8.4705882352941175e-01   9.4117647058823528e-01   8.2352941176470584e-01
           8.5098039215686272e-01   9.4117647058823528e-01   8.2745098039215681e-01
           8.5490196078431369e-01   9.4509803921568625e-01   8.3137254901960789e-01
           8.5882352941176465e-01   9.4509803921568625e-01   8.3529411764705885e-01
           8.6274509803921573e-01   9.4509803921568625e-01   8.3921568627450982e-01
           8.6666666666666670e-01   9.4901960784313721e-01   8.4313725490196079e-01
           8.7058823529411766e-01   9.4901960784313721e-01   8.4705882352941175e-01
           8.7450980392156863e-01   9.5294117647058818e-01   8.5098039215686272e-01
           8.7843137254901960e-01   9.5294117647058818e-01   8.5490196078431369e-01
           8.7843137254901960e-01   9.5294117647058818e-01   8.5882352941176465e-01
           8.8235294117647056e-01   9.5686274509803926e-01   8.6274509803921573e-01
           8.8627450980392153e-01   9.5686274509803926e-01   8.6666666666666670e-01
           8.9019607843137250e-01   9.5686274509803926e-01   8.7058823529411766e-01
           8.9411764705882357e-01   9.5686274509803926e-01   8.7058823529411766e-01
           8.9411764705882357e-01   9.6078431372549022e-01   8.7450980392156863e-01
           8.9803921568627454e-01   9.6078431372549022e-01   8.7843137254901960e-01
           9.0196078431372551e-01   9.6078431372549022e-01   8.8235294117647056e-01
           9.0196078431372551e-01   9.6470588235294119e-01   8.8627450980392153e-01
           9.0588235294117647e-01   9.6470588235294119e-01   8.8627450980392153e-01
           9.0980392156862744e-01   9.6470588235294119e-01   8.9019607843137250e-01
           9.0980392156862744e-01   9.6470588235294119e-01   8.9411764705882357e-01
           9.1372549019607840e-01   9.6862745098039216e-01   8.9803921568627454e-01
           9.1764705882352937e-01   9.6862745098039216e-01   8.9803921568627454e-01
           9.1764705882352937e-01   9.6862745098039216e-01   9.0196078431372551e-01
           9.2156862745098034e-01   9.6862745098039216e-01   9.0588235294117647e-01
           9.2549019607843142e-01   9.7254901960784312e-01   9.0980392156862744e-01
           9.2549019607843142e-01   9.7254901960784312e-01   9.0980392156862744e-01
           9.2941176470588238e-01   9.7254901960784312e-01   9.1372549019607840e-01
           9.3333333333333335e-01   9.7254901960784312e-01   9.1764705882352937e-01
           9.3333333333333335e-01   9.7647058823529409e-01   9.1764705882352937e-01
           9.3725490196078431e-01   9.7647058823529409e-01   9.2156862745098034e-01
           9.3725490196078431e-01   9.7647058823529409e-01   9.2549019607843142e-01
           9.4117647058823528e-01   9.7647058823529409e-01   9.2549019607843142e-01
           9.4117647058823528e-01   9.7647058823529409e-01   9.2941176470588238e-01
           9.4509803921568625e-01   9.8039215686274506e-01   9.3333333333333335e-01
           9.4901960784313721e-01   9.8039215686274506e-01   9.3333333333333335e-01
           9.4901960784313721e-01   9.8039215686274506e-01   9.3725490196078431e-01
           9.5294117647058818e-01   9.8039215686274506e-01   9.4117647058823528e-01
           9.5294117647058818e-01   9.8039215686274506e-01   9.4117647058823528e-01
           9.5686274509803926e-01   9.8431372549019602e-01   9.4509803921568625e-01
           9.5686274509803926e-01   9.8431372549019602e-01   9.4509803921568625e-01
           9.5686274509803926e-01   9.8431372549019602e-01   9.4901960784313721e-01
           9.6078431372549022e-01   9.8431372549019602e-01   9.4901960784313721e-01
           9.6078431372549022e-01   9.8431372549019602e-01   9.5294117647058818e-01
           9.6470588235294119e-01   9.8823529411764710e-01   9.5294117647058818e-01
           9.6470588235294119e-01   9.8823529411764710e-01   9.5686274509803926e-01
           9.6862745098039216e-01   9.8823529411764710e-01   9.6078431372549022e-01
           9.6862745098039216e-01   9.8823529411764710e-01   9.6078431372549022e-01 ];
    case 'diverging'
        Data=flipud([...
           4.0392156862745099e-01   0.0000000000000000e+00   1.2156862745098039e-01
           4.1960784313725491e-01   0.0000000000000000e+00   1.2156862745098039e-01
           4.3137254901960786e-01   0.0000000000000000e+00   1.2156862745098039e-01
           4.4705882352941179e-01   0.0000000000000000e+00   1.2156862745098039e-01
           4.5882352941176469e-01   3.9215686274509803e-03   1.2156862745098039e-01
           4.7450980392156861e-01   3.9215686274509803e-03   1.2549019607843137e-01
           4.8627450980392156e-01   7.8431372549019607e-03   1.2549019607843137e-01
           4.9803921568627452e-01   7.8431372549019607e-03   1.2549019607843137e-01
           5.1372549019607838e-01   1.1764705882352941e-02   1.2941176470588237e-01
           5.2549019607843139e-01   1.5686274509803921e-02   1.2941176470588237e-01
           5.3725490196078429e-01   1.9607843137254902e-02   1.2941176470588237e-01
           5.4901960784313730e-01   1.9607843137254902e-02   1.3333333333333333e-01
           5.6078431372549020e-01   2.3529411764705882e-02   1.3333333333333333e-01
           5.7254901960784310e-01   2.7450980392156862e-02   1.3725490196078433e-01
           5.8431372549019611e-01   3.5294117647058823e-02   1.3725490196078433e-01
           5.9607843137254901e-01   3.9215686274509803e-02   1.4117647058823529e-01
           6.0784313725490191e-01   4.3137254901960784e-02   1.4117647058823529e-01
           6.1568627450980395e-01   4.7058823529411764e-02   1.4509803921568629e-01
           6.2745098039215685e-01   5.0980392156862744e-02   1.4901960784313725e-01
           6.3529411764705879e-01   5.4901960784313725e-02   1.4901960784313725e-01
           6.4705882352941180e-01   6.2745098039215685e-02   1.5294117647058825e-01
           6.5490196078431373e-01   6.6666666666666666e-02   1.5686274509803921e-01
           6.6666666666666663e-01   7.0588235294117646e-02   1.5686274509803921e-01
           6.7450980392156867e-01   7.8431372549019607e-02   1.6078431372549021e-01
           6.8235294117647061e-01   8.2352941176470587e-02   1.6470588235294117e-01
           6.9019607843137254e-01   9.0196078431372548e-02   1.6470588235294117e-01
           6.9803921568627447e-01   9.4117647058823528e-02   1.6862745098039217e-01
           7.0588235294117652e-01   1.0196078431372549e-01   1.7254901960784313e-01
           7.1372549019607845e-01   1.0588235294117647e-01   1.7647058823529413e-01
           7.1764705882352942e-01   1.1372549019607843e-01   1.7647058823529413e-01
           7.2549019607843135e-01   1.2156862745098039e-01   1.8039215686274510e-01
           7.3333333333333328e-01   1.3333333333333333e-01   1.8431372549019609e-01
           7.3725490196078436e-01   1.4117647058823529e-01   1.9215686274509805e-01
           7.4509803921568629e-01   1.5294117647058825e-01   1.9607843137254902e-01
           7.5294117647058822e-01   1.6470588235294117e-01   2.0000000000000001e-01
           7.5686274509803919e-01   1.7647058823529413e-01   2.0392156862745098e-01
           7.6078431372549016e-01   1.8823529411764706e-01   2.1176470588235294e-01
           7.6862745098039220e-01   2.0000000000000001e-01   2.1568627450980393e-01
           7.7254901960784317e-01   2.1176470588235294e-01   2.1960784313725490e-01
           7.8039215686274510e-01   2.2352941176470589e-01   2.2745098039215686e-01
           7.8431372549019607e-01   2.3921568627450981e-01   2.3137254901960785e-01
           7.8823529411764703e-01   2.5098039215686274e-01   2.3921568627450981e-01
           7.9607843137254897e-01   2.6274509803921570e-01   2.4313725490196078e-01
           8.0000000000000004e-01   2.7843137254901962e-01   2.5098039215686274e-01
           8.0392156862745101e-01   2.9019607843137257e-01   2.5882352941176473e-01
           8.0784313725490198e-01   3.0588235294117649e-01   2.6274509803921570e-01
           8.1568627450980391e-01   3.1764705882352939e-01   2.7058823529411763e-01
           8.1960784313725488e-01   3.2941176470588235e-01   2.7450980392156865e-01
           8.2352941176470584e-01   3.4117647058823530e-01   2.8235294117647058e-01
           8.2745098039215681e-01   3.5294117647058826e-01   2.9019607843137257e-01
           8.3529411764705885e-01   3.6470588235294116e-01   2.9411764705882354e-01
           8.3921568627450982e-01   3.7647058823529411e-01   3.0196078431372547e-01
           8.4313725490196079e-01   3.8823529411764707e-01   3.0980392156862746e-01
           8.5098039215686272e-01   4.0000000000000002e-01   3.1372549019607843e-01
           8.5490196078431369e-01   4.0784313725490196e-01   3.2156862745098042e-01
           8.5882352941176465e-01   4.1960784313725491e-01   3.2941176470588235e-01
           8.6666666666666670e-01   4.3137254901960786e-01   3.3725490196078434e-01
           8.7058823529411766e-01   4.4313725490196076e-01   3.4117647058823530e-01
           8.7450980392156863e-01   4.5098039215686275e-01   3.4901960784313724e-01
           8.8235294117647056e-01   4.6274509803921571e-01   3.5686274509803922e-01
           8.8627450980392153e-01   4.7450980392156861e-01   3.6470588235294116e-01
           8.9019607843137250e-01   4.8627450980392156e-01   3.7254901960784315e-01
           8.9803921568627454e-01   4.9411764705882355e-01   3.8039215686274508e-01
           9.0196078431372551e-01   5.0588235294117645e-01   3.8823529411764707e-01
           9.0588235294117647e-01   5.1764705882352946e-01   3.9607843137254900e-01
           9.1372549019607840e-01   5.2549019607843139e-01   4.0392156862745099e-01
           9.1764705882352937e-01   5.3725490196078429e-01   4.1176470588235292e-01
           9.2156862745098034e-01   5.4901960784313730e-01   4.1960784313725491e-01
           9.2549019607843142e-01   5.5686274509803924e-01   4.3137254901960786e-01
           9.2941176470588238e-01   5.6862745098039214e-01   4.3921568627450980e-01
           9.3333333333333335e-01   5.8039215686274515e-01   4.4705882352941179e-01
           9.3725490196078431e-01   5.8823529411764708e-01   4.5490196078431372e-01
           9.4117647058823528e-01   5.9999999999999998e-01   4.6274509803921571e-01
           9.4509803921568625e-01   6.0784313725490191e-01   4.7450980392156861e-01
           9.4901960784313721e-01   6.1960784313725492e-01   4.8235294117647060e-01
           9.5294117647058818e-01   6.2745098039215685e-01   4.9019607843137253e-01
           9.5294117647058818e-01   6.3921568627450975e-01   5.0196078431372548e-01
           9.5686274509803926e-01   6.4705882352941180e-01   5.0980392156862742e-01
           9.6078431372549022e-01   6.5490196078431373e-01   5.1764705882352946e-01
           9.6078431372549022e-01   6.6666666666666663e-01   5.2941176470588236e-01
           9.6470588235294119e-01   6.7450980392156867e-01   5.3725490196078429e-01
           9.6470588235294119e-01   6.8627450980392157e-01   5.4901960784313730e-01
           9.6862745098039216e-01   6.9411764705882351e-01   5.6078431372549020e-01
           9.6862745098039216e-01   7.0588235294117652e-01   5.7254901960784310e-01
           9.7254901960784312e-01   7.1372549019607845e-01   5.8431372549019611e-01
           9.7254901960784312e-01   7.2156862745098038e-01   5.9607843137254901e-01
           9.7647058823529409e-01   7.3333333333333328e-01   6.0392156862745094e-01
           9.7647058823529409e-01   7.4117647058823533e-01   6.1568627450980395e-01
           9.8039215686274506e-01   7.4901960784313726e-01   6.2745098039215685e-01
           9.8039215686274506e-01   7.6078431372549016e-01   6.3921568627450975e-01
           9.8039215686274506e-01   7.6862745098039220e-01   6.5098039215686276e-01
           9.8431372549019602e-01   7.7647058823529413e-01   6.6274509803921566e-01
           9.8431372549019602e-01   7.8823529411764703e-01   6.7450980392156867e-01
           9.8431372549019602e-01   7.9607843137254897e-01   6.8627450980392157e-01
           9.8823529411764710e-01   8.0392156862745101e-01   6.9803921568627447e-01
           9.8823529411764710e-01   8.1176470588235294e-01   7.0980392156862748e-01
           9.8823529411764710e-01   8.1960784313725488e-01   7.2156862745098038e-01
           9.9215686274509807e-01   8.2745098039215681e-01   7.3333333333333328e-01
           9.9215686274509807e-01   8.3529411764705885e-01   7.4117647058823533e-01
           9.9215686274509807e-01   8.3921568627450982e-01   7.5294117647058822e-01
           9.9215686274509807e-01   8.4705882352941175e-01   7.6078431372549016e-01
           9.9215686274509807e-01   8.5490196078431369e-01   7.7254901960784317e-01
           9.9215686274509807e-01   8.5882352941176465e-01   7.8039215686274510e-01
           9.9215686274509807e-01   8.6274509803921573e-01   7.8823529411764703e-01
           9.9215686274509807e-01   8.7058823529411766e-01   8.0000000000000004e-01
           9.9215686274509807e-01   8.7450980392156863e-01   8.0784313725490198e-01
           9.9215686274509807e-01   8.8235294117647056e-01   8.1568627450980391e-01
           9.9215686274509807e-01   8.8627450980392153e-01   8.2745098039215681e-01
           9.9215686274509807e-01   8.9411764705882357e-01   8.3529411764705885e-01
           9.8823529411764710e-01   8.9803921568627454e-01   8.4705882352941175e-01
           9.8823529411764710e-01   9.0588235294117647e-01   8.5490196078431369e-01
           9.8823529411764710e-01   9.0980392156862744e-01   8.6666666666666670e-01
           9.8823529411764710e-01   9.1764705882352937e-01   8.7450980392156863e-01
           9.8823529411764710e-01   9.2156862745098034e-01   8.8627450980392153e-01
           9.8823529411764710e-01   9.2549019607843142e-01   8.9411764705882357e-01
           9.8431372549019602e-01   9.3333333333333335e-01   9.0196078431372551e-01
           9.8431372549019602e-01   9.3725490196078431e-01   9.0980392156862744e-01
           9.8431372549019602e-01   9.4117647058823528e-01   9.1764705882352937e-01
           9.8431372549019602e-01   9.4509803921568625e-01   9.2549019607843142e-01
           9.8039215686274506e-01   9.4901960784313721e-01   9.3333333333333335e-01
           9.8039215686274506e-01   9.5294117647058818e-01   9.4117647058823528e-01
           9.8039215686274506e-01   9.5686274509803926e-01   9.4509803921568625e-01
           9.7647058823529409e-01   9.6078431372549022e-01   9.5294117647058818e-01
           9.7647058823529409e-01   9.6078431372549022e-01   9.5686274509803926e-01
           9.7647058823529409e-01   9.6470588235294119e-01   9.6078431372549022e-01
           9.7254901960784312e-01   9.6470588235294119e-01   9.6470588235294119e-01
           9.7254901960784312e-01   9.6862745098039216e-01   9.6862745098039216e-01
           9.6862745098039216e-01   9.6862745098039216e-01   9.6862745098039216e-01
           9.6862745098039216e-01   9.6862745098039216e-01   9.6862745098039216e-01
           9.6862745098039216e-01   9.6862745098039216e-01   9.6862745098039216e-01
           9.6470588235294119e-01   9.6862745098039216e-01   9.6862745098039216e-01
           9.6078431372549022e-01   9.6862745098039216e-01   9.6862745098039216e-01
           9.5686274509803926e-01   9.6470588235294119e-01   9.6862745098039216e-01
           9.5294117647058818e-01   9.6470588235294119e-01   9.6862745098039216e-01
           9.4901960784313721e-01   9.6078431372549022e-01   9.6470588235294119e-01
           9.4509803921568625e-01   9.6078431372549022e-01   9.6470588235294119e-01
           9.4117647058823528e-01   9.5686274509803926e-01   9.6470588235294119e-01
           9.3333333333333335e-01   9.5686274509803926e-01   9.6470588235294119e-01
           9.2941176470588238e-01   9.5294117647058818e-01   9.6078431372549022e-01
           9.2156862745098034e-01   9.4901960784313721e-01   9.6078431372549022e-01
           9.1372549019607840e-01   9.4509803921568625e-01   9.6078431372549022e-01
           9.0980392156862744e-01   9.4117647058823528e-01   9.6078431372549022e-01
           9.0196078431372551e-01   9.4117647058823528e-01   9.5686274509803926e-01
           8.9411764705882357e-01   9.3725490196078431e-01   9.5686274509803926e-01
           8.8627450980392153e-01   9.3333333333333335e-01   9.5686274509803926e-01
           8.7843137254901960e-01   9.2941176470588238e-01   9.5294117647058818e-01
           8.7058823529411766e-01   9.2549019607843142e-01   9.5294117647058818e-01
           8.6666666666666670e-01   9.2156862745098034e-01   9.4901960784313721e-01
           8.5882352941176465e-01   9.1764705882352937e-01   9.4901960784313721e-01
           8.5098039215686272e-01   9.1372549019607840e-01   9.4901960784313721e-01
           8.4313725490196079e-01   9.0980392156862744e-01   9.4509803921568625e-01
           8.3529411764705885e-01   9.0588235294117647e-01   9.4509803921568625e-01
           8.2745098039215681e-01   9.0196078431372551e-01   9.4117647058823528e-01
           8.1960784313725488e-01   8.9803921568627454e-01   9.4117647058823528e-01
           8.1176470588235294e-01   8.9411764705882357e-01   9.4117647058823528e-01
           8.0392156862745101e-01   8.9019607843137250e-01   9.3725490196078431e-01
           7.9607843137254897e-01   8.8627450980392153e-01   9.3725490196078431e-01
           7.8823529411764703e-01   8.8235294117647056e-01   9.3333333333333335e-01
           7.8039215686274510e-01   8.7843137254901960e-01   9.3333333333333335e-01
           7.7254901960784317e-01   8.7450980392156863e-01   9.2941176470588238e-01
           7.6470588235294112e-01   8.7058823529411766e-01   9.2549019607843142e-01
           7.5294117647058822e-01   8.6666666666666670e-01   9.2549019607843142e-01
           7.4509803921568629e-01   8.6274509803921573e-01   9.2156862745098034e-01
           7.3725490196078436e-01   8.5882352941176465e-01   9.2156862745098034e-01
           7.2549019607843135e-01   8.5098039215686272e-01   9.1764705882352937e-01
           7.1764705882352942e-01   8.4705882352941175e-01   9.1372549019607840e-01
           7.0588235294117652e-01   8.4313725490196079e-01   9.0980392156862744e-01
           6.9803921568627447e-01   8.3921568627450982e-01   9.0980392156862744e-01
           6.8627450980392157e-01   8.3137254901960789e-01   9.0588235294117647e-01
           6.7843137254901964e-01   8.2745098039215681e-01   9.0196078431372551e-01
           6.6666666666666663e-01   8.2352941176470584e-01   8.9803921568627454e-01
           6.5882352941176470e-01   8.1960784313725488e-01   8.9803921568627454e-01
           6.4705882352941180e-01   8.1176470588235294e-01   8.9411764705882357e-01
           6.3529411764705879e-01   8.0784313725490198e-01   8.9019607843137250e-01
           6.2745098039215685e-01   8.0000000000000004e-01   8.8627450980392153e-01
           6.1568627450980395e-01   7.9607843137254897e-01   8.8235294117647056e-01
           6.0392156862745094e-01   7.8823529411764703e-01   8.8235294117647056e-01
           5.9215686274509804e-01   7.8431372549019607e-01   8.7843137254901960e-01
           5.8431372549019611e-01   7.8039215686274510e-01   8.7450980392156863e-01
           5.7254901960784310e-01   7.7254901960784317e-01   8.7058823529411766e-01
           5.6078431372549020e-01   7.6470588235294112e-01   8.6666666666666670e-01
           5.4901960784313730e-01   7.6078431372549016e-01   8.6274509803921573e-01
           5.3725490196078429e-01   7.5294117647058822e-01   8.5882352941176465e-01
           5.2549019607843139e-01   7.4509803921568629e-01   8.5490196078431369e-01
           5.0980392156862742e-01   7.3725490196078436e-01   8.5098039215686272e-01
           4.9803921568627452e-01   7.2941176470588232e-01   8.4705882352941175e-01
           4.8627450980392156e-01   7.2549019607843135e-01   8.4313725490196079e-01
           4.7058823529411764e-01   7.1764705882352942e-01   8.3921568627450982e-01
           4.5490196078431372e-01   7.0588235294117652e-01   8.3529411764705885e-01
           4.4313725490196076e-01   6.9803921568627447e-01   8.3137254901960789e-01
           4.2745098039215684e-01   6.9019607843137254e-01   8.2745098039215681e-01
           4.1568627450980394e-01   6.8235294117647061e-01   8.1960784313725488e-01
           4.0000000000000002e-01   6.7450980392156867e-01   8.1568627450980391e-01
           3.8431372549019610e-01   6.6666666666666663e-01   8.1176470588235294e-01
           3.7254901960784315e-01   6.5882352941176470e-01   8.0784313725490198e-01
           3.5686274509803922e-01   6.4705882352941180e-01   8.0392156862745101e-01
           3.4509803921568627e-01   6.3921568627450975e-01   8.0000000000000004e-01
           3.3333333333333331e-01   6.3137254901960782e-01   7.9607843137254897e-01
           3.2156862745098042e-01   6.2352941176470589e-01   7.8823529411764703e-01
           3.0980392156862746e-01   6.1568627450980395e-01   7.8431372549019607e-01
           2.9803921568627451e-01   6.0784313725490191e-01   7.8039215686274510e-01
           2.9019607843137257e-01   5.9999999999999998e-01   7.7647058823529413e-01
           2.7843137254901962e-01   5.9215686274509804e-01   7.7254901960784317e-01
           2.7058823529411763e-01   5.8431372549019611e-01   7.6862745098039220e-01
           2.6274509803921570e-01   5.7647058823529407e-01   7.6470588235294112e-01
           2.5490196078431371e-01   5.6862745098039214e-01   7.6078431372549016e-01
           2.4705882352941178e-01   5.6078431372549020e-01   7.5686274509803919e-01
           2.4313725490196078e-01   5.5686274509803924e-01   7.5294117647058822e-01
           2.3529411764705882e-01   5.4901960784313730e-01   7.4901960784313726e-01
           2.3137254901960785e-01   5.4117647058823526e-01   7.4901960784313726e-01
           2.2352941176470589e-01   5.3333333333333333e-01   7.4509803921568629e-01
           2.1960784313725490e-01   5.2941176470588236e-01   7.4117647058823533e-01
           2.1176470588235294e-01   5.2156862745098043e-01   7.3725490196078436e-01
           2.0784313725490197e-01   5.1372549019607838e-01   7.3725490196078436e-01
           2.0392156862745098e-01   5.0980392156862742e-01   7.3333333333333328e-01
           1.9607843137254902e-01   5.0196078431372548e-01   7.2941176470588232e-01
           1.9215686274509805e-01   4.9411764705882355e-01   7.2941176470588232e-01
           1.8823529411764706e-01   4.9019607843137253e-01   7.2549019607843135e-01
           1.8431372549019609e-01   4.8235294117647060e-01   7.2156862745098038e-01
           1.8039215686274510e-01   4.7450980392156861e-01   7.1764705882352942e-01
           1.7254901960784313e-01   4.7058823529411764e-01   7.1372549019607845e-01
           1.6862745098039217e-01   4.6274509803921571e-01   7.1372549019607845e-01
           1.6470588235294117e-01   4.5490196078431372e-01   7.0980392156862748e-01
           1.6078431372549021e-01   4.5098039215686275e-01   7.0588235294117652e-01
           1.5686274509803921e-01   4.4313725490196076e-01   7.0196078431372544e-01
           1.5294117647058825e-01   4.3529411764705883e-01   6.9803921568627447e-01
           1.4901960784313725e-01   4.2745098039215684e-01   6.9411764705882351e-01
           1.4509803921568629e-01   4.2352941176470588e-01   6.9019607843137254e-01
           1.3725490196078433e-01   4.1568627450980394e-01   6.8627450980392157e-01
           1.3333333333333333e-01   4.0784313725490196e-01   6.7843137254901964e-01
           1.2941176470588237e-01   4.0000000000000002e-01   6.7450980392156867e-01
           1.2549019607843137e-01   3.9215686274509803e-01   6.7058823529411760e-01
           1.2156862745098039e-01   3.8431372549019610e-01   6.6274509803921566e-01
           1.1372549019607843e-01   3.7647058823529411e-01   6.5490196078431373e-01
           1.0980392156862745e-01   3.6862745098039218e-01   6.4705882352941180e-01
           1.0588235294117647e-01   3.6078431372549019e-01   6.3921568627450975e-01
           1.0196078431372549e-01   3.5294117647058826e-01   6.3137254901960782e-01
           9.8039215686274508e-02   3.4509803921568627e-01   6.1960784313725492e-01
           9.4117647058823528e-02   3.3725490196078434e-01   6.1176470588235299e-01
           8.6274509803921567e-02   3.2941176470588235e-01   5.9999999999999998e-01
           8.2352941176470587e-02   3.2156862745098042e-01   5.8823529411764708e-01
           7.8431372549019607e-02   3.1372549019607843e-01   5.7647058823529407e-01
           7.4509803921568626e-02   3.0196078431372547e-01   5.6470588235294117e-01
           7.0588235294117646e-02   2.9411764705882354e-01   5.5294117647058827e-01
           6.6666666666666666e-02   2.8627450980392155e-01   5.4117647058823526e-01
           6.2745098039215685e-02   2.7843137254901962e-01   5.2941176470588236e-01
           5.8823529411764705e-02   2.7058823529411763e-01   5.1372549019607838e-01
           5.0980392156862744e-02   2.5882352941176473e-01   5.0196078431372548e-01
           4.7058823529411764e-02   2.5098039215686274e-01   4.8627450980392156e-01
           4.3137254901960784e-02   2.4313725490196078e-01   4.7058823529411764e-01
           3.9215686274509803e-02   2.3529411764705882e-01   4.5882352941176469e-01
           3.5294117647058823e-02   2.2352941176470589e-01   4.4313725490196076e-01
           3.1372549019607843e-02   2.1568627450980393e-01   4.2745098039215684e-01
           2.7450980392156862e-02   2.0784313725490197e-01   4.1176470588235292e-01
           2.3529411764705882e-02   1.9607843137254902e-01   3.9607843137254900e-01
           1.9607843137254902e-02   1.8823529411764706e-01   3.8039215686274508e-01]);
    case 'land'
        % Land colours from ETOPO1
        Data=[          
            0.56410 0.20000 0.40000 0.00000
            0.56920 0.20000 0.80000 0.40000
            0.57440 0.73333 0.89412 0.57255
            0.58970 1.00000 0.86275 0.72549
            0.61540 0.95294 0.79216 0.53725
            0.64100 0.90196 0.72157 0.34510
            0.66670 0.85098 0.65098 0.15294
            0.69230 0.65882 0.60392 0.12157
            0.71790 0.64314 0.56471 0.09804
            0.74360 0.63529 0.52549 0.07451
            0.76920 0.62353 0.48235 0.05098
            0.79490 0.61176 0.44314 0.02745
            0.82050 0.60000 0.40000 0.00000
            0.84620 0.63529 0.34902 0.34902
            0.87180 0.69804 0.46275 0.46275
            0.89740 0.71765 0.57647 0.57647
            0.92310 0.76078 0.69020 0.69020
            0.94870 0.80000 0.80000 0.80000
            0.97440 0.89804 0.89804 0.89804
            1.00000 1.00000 1.00000 1.00000];
        r=(Data(:,1)-Data(1,1))/(Data(end,1)-Data(1,1))*255;    
        Data=Data(:,2:4);
    case 'water'
        % Water colours from ETOPO1
        Data=[0.00000 0.03922 0.00000 0.47451
            0.02560 0.10196 0.00000 0.53725
            0.05130 0.14902 0.00000 0.59608
            0.07690 0.10588 0.01176 0.65098
            0.10260 0.06275 0.02353 0.70588
            0.12820 0.01961 0.03529 0.75686
            0.15380 0.00000 0.05490 0.79608
            0.17950 0.00000 0.08627 0.82353
            0.20510 0.00000 0.11765 0.84706
            0.23080 0.00000 0.15294 0.87451
            0.25640 0.04706 0.26667 0.90588
            0.28210 0.10196 0.40000 0.94118
            0.30770 0.07451 0.45882 0.95686
            0.33330 0.05490 0.52157 0.97647
            0.35900 0.08235 0.61961 0.98824
            0.38460 0.11765 0.69804 1.00000
            0.41030 0.16863 0.72941 1.00000
            0.43590 0.21569 0.75686 1.00000
            0.46150 0.25490 0.78431 1.00000
            0.48720 0.30980 0.82353 1.00000
            0.51280 0.36863 0.87451 1.00000
            0.53850 0.54118 0.89020 1.00000
            0.56410 0.73725 0.90196 1.00000];
        r=(Data(:,1)-Data(1,1))/(Data(end,1)-Data(1,1))*255;    
        Data=Data(:,2:4);
    case 'gland'
        % Land colours from ETOPO1
        Data=[          
            0.56410 0.20000 0.40000 0.00000
            0.56920 0.20000 0.80000 0.40000
            0.57440 0.73333 0.89412 0.57255
            0.58970 1.00000 0.86275 0.72549
            0.61540 0.95294 0.79216 0.53725
            0.64100 0.90196 0.72157 0.34510
            0.66670 0.85098 0.65098 0.15294
            0.69230 0.65882 0.60392 0.12157
            0.71790 0.64314 0.56471 0.09804
            0.74360 0.63529 0.52549 0.07451
            0.76920 0.62353 0.48235 0.05098
            0.79490 0.61176 0.44314 0.02745
            0.82050 0.60000 0.40000 0.00000
            0.84620 0.63529 0.34902 0.34902
            0.87180 0.69804 0.46275 0.46275
            0.89740 0.71765 0.57647 0.57647
            0.92310 0.76078 0.69020 0.69020
            0.94870 0.80000 0.80000 0.80000
            0.97440 0.89804 0.89804 0.89804
            1.00000 1.00000 1.00000 1.00000];
        % Same colours as in 'land' but I stretch the
        % green a little
        r=[0:length(Data)-1]/(length(Data)-1)*255;
        Data=Data(:,2:4); 
    case 'bland'
        % Land colours from ETOPO1
        Data=[          
          %  0.56410 0.20000 0.40000 0.00000
          %  0.56920 0.20000 0.80000 0.40000
          %  0.57440 0.73333 0.89412 0.57255
            0.58970 1.00000 0.86275 0.72549
            0.61540 0.95294 0.79216 0.53725
            0.64100 0.90196 0.72157 0.34510
            0.66670 0.85098 0.65098 0.15294
            0.69230 0.65882 0.60392 0.12157
            0.71790 0.64314 0.56471 0.09804
            0.74360 0.63529 0.52549 0.07451
            0.76920 0.62353 0.48235 0.05098
            0.79490 0.61176 0.44314 0.02745
            0.82050 0.60000 0.40000 0.00000
            0.84620 0.63529 0.34902 0.34902
            0.87180 0.69804 0.46275 0.46275
            0.89740 0.71765 0.57647 0.57647
            0.92310 0.76078 0.69020 0.69020
            0.94870 0.80000 0.80000 0.80000
            0.97440 0.89804 0.89804 0.89804
            1.00000 1.00000 1.00000 1.00000];
        % Same colours as in 'land' but I remove the
        % Green
        r=[0:length(Data)-1]/(length(Data)-1)*255;
        Data=Data(:,2:4); 
    case 'odv'
        % Created from CET toolbox.
        Data=[ 9.0101331e-01   0.0000000e+00   9.3251202e-01
               8.9241369e-01   5.4084707e-02   9.3493161e-01
               8.8374697e-01   9.2009868e-02   9.3733662e-01
               8.7501086e-01   1.1940643e-01   9.3972123e-01
               8.6621022e-01   1.4173077e-01   9.4208765e-01
               8.5733587e-01   1.6085164e-01   9.4443514e-01
               8.4839234e-01   1.7779733e-01   9.4676122e-01
               8.3937643e-01   1.9304161e-01   9.4906833e-01
               8.3028040e-01   2.0703619e-01   9.5135665e-01
               8.2110339e-01   2.1996946e-01   9.5362260e-01
               8.1184800e-01   2.3203951e-01   9.5586460e-01
               8.0250437e-01   2.4334747e-01   9.5808151e-01
               7.9308093e-01   2.5406699e-01   9.6027147e-01
               7.8356163e-01   2.6421451e-01   9.6243395e-01
               7.7394557e-01   2.7387323e-01   9.6457095e-01
               7.6424053e-01   2.8309721e-01   9.6668309e-01
               7.5443764e-01   2.9192664e-01   9.6876443e-01
               7.4452229e-01   3.0039475e-01   9.7081305e-01
               7.3450848e-01   3.0856348e-01   9.7283330e-01
               7.2437830e-01   3.1642474e-01   9.7481118e-01
               7.1413293e-01   3.2400124e-01   9.7675955e-01
               7.0377315e-01   3.3133238e-01   9.7866788e-01
               6.9328911e-01   3.3844029e-01   9.8054235e-01
               6.8267129e-01   3.4532943e-01   9.8237525e-01
               6.7192363e-01   3.5200255e-01   9.8416286e-01
               6.6103315e-01   3.5849315e-01   9.8590542e-01
               6.5000382e-01   3.6480311e-01   9.8760208e-01
               6.3881994e-01   3.7093737e-01   9.8924507e-01
               6.2748126e-01   3.7692778e-01   9.9083128e-01
               6.1597716e-01   3.8274509e-01   9.9236641e-01
               6.0430713e-01   3.8842528e-01   9.9384206e-01
               5.9245905e-01   3.9395999e-01   9.9524582e-01
               5.8041851e-01   3.9936053e-01   9.9658429e-01
               5.6818292e-01   4.0464332e-01   9.9785650e-01
               5.5574776e-01   4.0981228e-01   9.9904764e-01
               5.4311794e-01   4.1484604e-01   1.0000000e+00
               5.3025652e-01   4.1977665e-01   1.0000000e+00
               5.1717467e-01   4.2459106e-01   1.0000000e+00
               5.0384831e-01   4.2929918e-01   1.0000000e+00
               4.9028115e-01   4.3389560e-01   1.0000000e+00
               4.7646435e-01   4.3840653e-01   1.0000000e+00
               4.6239168e-01   4.4281244e-01   1.0000000e+00
               4.4807735e-01   4.4710328e-01   1.0000000e+00
               4.3348133e-01   4.5131439e-01   1.0000000e+00
               4.1882644e-01   4.5541169e-01   1.0000000e+00
               4.0463168e-01   4.5935111e-01   1.0000000e+00
               3.9127493e-01   4.6308440e-01   1.0000000e+00
               3.7887635e-01   4.6663821e-01   1.0000000e+00
               3.6754491e-01   4.6997659e-01   9.9710824e-01
               3.5727105e-01   4.7310156e-01   9.9361634e-01
               3.4806754e-01   4.7605687e-01   9.8960956e-01
               3.3991294e-01   4.7882200e-01   9.8516720e-01
               3.3278841e-01   4.8142044e-01   9.8033564e-01
               3.2657245e-01   4.8386769e-01   9.7517562e-01
               3.2131863e-01   4.8617771e-01   9.6972157e-01
               3.1692286e-01   4.8836273e-01   9.6400671e-01
               3.1327820e-01   4.9042788e-01   9.5808298e-01
               3.1036518e-01   4.9237894e-01   9.5195268e-01
               3.0812518e-01   4.9424286e-01   9.4564700e-01
               3.0652684e-01   4.9600072e-01   9.3918421e-01
               3.0545194e-01   4.9770307e-01   9.3258337e-01
               3.0485773e-01   4.9930273e-01   9.2586855e-01
               3.0471500e-01   5.0085383e-01   9.1903664e-01
               3.0497614e-01   5.0233287e-01   9.1211284e-01
               3.0559984e-01   5.0375514e-01   9.0509893e-01
               3.0652667e-01   5.0512067e-01   8.9799743e-01
               3.0771336e-01   5.0643775e-01   8.9082591e-01
               3.0918805e-01   5.0771753e-01   8.8359036e-01
               3.1084258e-01   5.0894669e-01   8.7628822e-01
               3.1265990e-01   5.1012895e-01   8.6893964e-01
               3.1464971e-01   5.1127898e-01   8.6153178e-01
               3.1677717e-01   5.1239777e-01   8.5407999e-01
               3.1899621e-01   5.1348864e-01   8.4658877e-01
               3.2129859e-01   5.1454679e-01   8.3906246e-01
               3.2370068e-01   5.1556074e-01   8.3149691e-01
               3.2615709e-01   5.1656269e-01   8.2388686e-01
               3.2867278e-01   5.1754427e-01   8.1625836e-01
               3.3117878e-01   5.1849579e-01   8.0859723e-01
               3.3372872e-01   5.1942240e-01   8.0091115e-01
               3.3627834e-01   5.2032367e-01   7.9320222e-01
               3.3885910e-01   5.2121527e-01   7.8546403e-01
               3.4141138e-01   5.2207962e-01   7.7770923e-01
               3.4392756e-01   5.2294489e-01   7.6993203e-01
               3.4645715e-01   5.2377972e-01   7.6213677e-01
               3.4896939e-01   5.2459818e-01   7.5432975e-01
               3.5143015e-01   5.2540630e-01   7.4649947e-01
               3.5384400e-01   5.2620510e-01   7.3866259e-01
               3.5626705e-01   5.2698442e-01   7.3080018e-01
               3.5861057e-01   5.2774386e-01   7.2292779e-01
               3.6090556e-01   5.2850832e-01   7.1504566e-01
               3.6318916e-01   5.2925382e-01   7.0715015e-01
               3.6541282e-01   5.2998346e-01   6.9923396e-01
               3.6755212e-01   5.3070830e-01   6.9133423e-01
               3.6961736e-01   5.3143077e-01   6.8342411e-01
               3.7153984e-01   5.3216045e-01   6.7551926e-01
               3.7333964e-01   5.3289069e-01   6.6762844e-01
               3.7498875e-01   5.3361944e-01   6.5974393e-01
               3.7651860e-01   5.3436635e-01   6.5187458e-01
               3.7790211e-01   5.3511255e-01   6.4400546e-01
               3.7916289e-01   5.3586250e-01   6.3616246e-01
               3.8030058e-01   5.3663010e-01   6.2832375e-01
               3.8132032e-01   5.3739134e-01   6.2049557e-01
               3.8220248e-01   5.3815863e-01   6.1266534e-01
               3.8298164e-01   5.3892674e-01   6.0486256e-01
               3.8363602e-01   5.3969563e-01   5.9706617e-01
               3.8417043e-01   5.4048220e-01   5.8926622e-01
               3.8459090e-01   5.4126271e-01   5.8148982e-01
               3.8489570e-01   5.4205784e-01   5.7372243e-01
               3.8508983e-01   5.4286004e-01   5.6595990e-01
               3.8517497e-01   5.4365697e-01   5.5820324e-01
               3.8514825e-01   5.4446264e-01   5.5046654e-01
               3.8500591e-01   5.4527490e-01   5.4273583e-01
               3.8475369e-01   5.4609572e-01   5.3500496e-01
               3.8439432e-01   5.4691158e-01   5.2728974e-01
               3.8392969e-01   5.4773483e-01   5.1957424e-01
               3.8335370e-01   5.4857211e-01   5.1186858e-01
               3.8264961e-01   5.4940346e-01   5.0418316e-01
               3.8184174e-01   5.5024295e-01   4.9647988e-01
               3.8093456e-01   5.5108859e-01   4.8880378e-01
               3.7989709e-01   5.5194307e-01   4.8112245e-01
               3.7875488e-01   5.5279313e-01   4.7344838e-01
               3.7750149e-01   5.5365179e-01   4.6578241e-01
               3.7613365e-01   5.5452404e-01   4.5812639e-01
               3.7463906e-01   5.5538969e-01   4.5047471e-01
               3.7301856e-01   5.5626729e-01   4.4282654e-01
               3.7128174e-01   5.5716409e-01   4.3518225e-01
               3.6943191e-01   5.5804236e-01   4.2753741e-01
               3.6744351e-01   5.5893816e-01   4.1989061e-01
               3.6533681e-01   5.5984781e-01   4.1225487e-01
               3.6309063e-01   5.6075086e-01   4.0460539e-01
               3.6070294e-01   5.6167024e-01   3.9696864e-01
               3.5819517e-01   5.6259588e-01   3.8934301e-01
               3.5553881e-01   5.6352351e-01   3.8170805e-01
               3.5271499e-01   5.6445380e-01   3.7408098e-01
               3.4974311e-01   5.6539030e-01   3.6644835e-01
               3.4660653e-01   5.6634992e-01   3.5879830e-01
               3.4330675e-01   5.6731182e-01   3.5117765e-01
               3.3982229e-01   5.6827515e-01   3.4353893e-01
               3.3613030e-01   5.6925183e-01   3.3590528e-01
               3.3226927e-01   5.7023982e-01   3.2827190e-01
               3.2817437e-01   5.7123976e-01   3.2064919e-01
               3.2386167e-01   5.7225439e-01   3.1300205e-01
               3.1930992e-01   5.7327706e-01   3.0538599e-01
               3.1448277e-01   5.7430284e-01   2.9775044e-01
               3.0940512e-01   5.7535700e-01   2.9013069e-01
               3.0397213e-01   5.7642766e-01   2.8255045e-01
               2.9821634e-01   5.7749511e-01   2.7494949e-01
               2.9213635e-01   5.7858904e-01   2.6736545e-01
               2.8585508e-01   5.7968241e-01   2.5973929e-01
               2.7979429e-01   5.8071733e-01   2.5182584e-01
               2.7409306e-01   5.8169092e-01   2.4353615e-01
               2.6895355e-01   5.8258780e-01   2.3488952e-01
               2.6452586e-01   5.8339123e-01   2.2571999e-01
               2.6115049e-01   5.8407359e-01   2.1600397e-01
               2.5921386e-01   5.8459763e-01   2.0562062e-01
               2.5917881e-01   5.8491193e-01   1.9457571e-01
               2.6172691e-01   5.8493530e-01   1.8281914e-01
               2.6736292e-01   5.8459009e-01   1.7079381e-01
               2.7605556e-01   5.8384546e-01   1.5913201e-01
               2.8684229e-01   5.8275563e-01   1.4851063e-01
               2.9881924e-01   5.8141442e-01   1.3929985e-01
               3.1118650e-01   5.7990916e-01   1.3150590e-01
               3.2354119e-01   5.7827384e-01   1.2482607e-01
               3.3571076e-01   5.7657878e-01   1.1919225e-01
               3.4760451e-01   5.7479693e-01   1.1436156e-01
               3.5917972e-01   5.7298881e-01   1.1035678e-01
               3.7046210e-01   5.7112054e-01   1.0690929e-01
               3.8146525e-01   5.6922129e-01   1.0397365e-01
               3.9217543e-01   5.6729630e-01   1.0155055e-01
               4.0262874e-01   5.6532047e-01   9.9563720e-02
               4.1285471e-01   5.6333643e-01   9.8014331e-02
               4.2283397e-01   5.6130090e-01   9.6695447e-02
               4.3264027e-01   5.5924194e-01   9.5762755e-02
               4.4224318e-01   5.5716236e-01   9.4951662e-02
               4.5166986e-01   5.5503657e-01   9.4080622e-02
               4.6093550e-01   5.5288878e-01   9.3190838e-02
               4.7006969e-01   5.5071469e-01   9.2394545e-02
               4.7905191e-01   5.4851288e-01   9.1613212e-02
               4.8791547e-01   5.4627320e-01   9.0767770e-02
               4.9663686e-01   5.4399494e-01   8.9948701e-02
               5.0525765e-01   5.4168472e-01   8.9142595e-02
               5.1377978e-01   5.3935013e-01   8.8344290e-02
               5.2217703e-01   5.3698984e-01   8.7579467e-02
               5.3049564e-01   5.3457717e-01   8.6781980e-02
               5.3871960e-01   5.3213942e-01   8.5936085e-02
               5.4684834e-01   5.2966356e-01   8.5197765e-02
               5.5489406e-01   5.2715147e-01   8.4499111e-02
               5.6286931e-01   5.2459702e-01   8.3736580e-02
               5.7075371e-01   5.2200153e-01   8.2965076e-02
               5.7857766e-01   5.1938381e-01   8.2294444e-02
               5.8633451e-01   5.1671589e-01   8.1575183e-02
               5.9402116e-01   5.1401457e-01   8.0830769e-02
               6.0164481e-01   5.1125679e-01   8.0119227e-02
               6.0921718e-01   5.0847758e-01   7.9421412e-02
               6.1672372e-01   5.0562705e-01   7.8739648e-02
               6.2417702e-01   5.0275308e-01   7.8075667e-02
               6.3157495e-01   4.9982760e-01   7.7428325e-02
               6.3892277e-01   4.9686181e-01   7.6797687e-02
               6.4622234e-01   4.9384916e-01   7.6184872e-02
               6.5347472e-01   4.9079302e-01   7.5589796e-02
               6.6068297e-01   4.8767582e-01   7.5007376e-02
               6.6785657e-01   4.8449686e-01   7.4444887e-02
               6.7497536e-01   4.8128060e-01   7.3917039e-02
               6.8205540e-01   4.7801106e-01   7.3382464e-02
               6.8910997e-01   4.7469164e-01   7.2799197e-02
               6.9611556e-01   4.7130644e-01   7.2248466e-02
               7.0308682e-01   4.6785815e-01   7.1781405e-02
               7.1002283e-01   4.6436799e-01   7.1317804e-02
               7.1692989e-01   4.6078972e-01   7.0812391e-02
               7.2379432e-01   4.5716541e-01   7.0334205e-02
               7.3063604e-01   4.5347355e-01   6.9928066e-02
               7.3744490e-01   4.4971842e-01   6.9550523e-02
               7.4422586e-01   4.4588089e-01   6.9158288e-02
               7.5098360e-01   4.4199555e-01   6.8753605e-02
               7.5770386e-01   4.3801889e-01   6.8363033e-02
               7.6440159e-01   4.3395308e-01   6.8000452e-02
               7.7107741e-01   4.2983143e-01   6.7662807e-02
               7.7772504e-01   4.2561806e-01   6.7345763e-02
               7.8435064e-01   4.2131316e-01   6.7047765e-02
               7.9094561e-01   4.1693930e-01   6.6769501e-02
               7.9752256e-01   4.1246414e-01   6.6511686e-02
               8.0408218e-01   4.0788143e-01   6.6274748e-02
               8.1061283e-01   4.0321333e-01   6.6058803e-02
               8.1712938e-01   3.9843601e-01   6.5863871e-02
               8.2362256e-01   3.9354671e-01   6.5689946e-02
               8.3010058e-01   3.8855623e-01   6.5537030e-02
               8.3655552e-01   3.8344453e-01   6.5405124e-02
               8.4299328e-01   3.7819900e-01   6.5294161e-02
               8.4941547e-01   3.7282716e-01   6.5204178e-02
               8.5581779e-01   3.6732198e-01   6.5135120e-02
               8.6220650e-01   3.6168376e-01   6.5087154e-02
               8.6857747e-01   3.5588751e-01   6.5060182e-02
               8.7492597e-01   3.4992373e-01   6.5054456e-02
               8.8126652e-01   3.4378373e-01   6.5069774e-02
               8.8758708e-01   3.3748220e-01   6.5106460e-02
               8.9389887e-01   3.3096597e-01   6.5164196e-02
               9.0019487e-01   3.2426128e-01   6.5243333e-02
               9.0647723e-01   3.1733910e-01   6.5343657e-02
               9.1274449e-01   3.1018523e-01   6.5465418e-02
               9.1899599e-01   3.0274206e-01   6.5608630e-02
               9.2524045e-01   2.9504892e-01   6.5773425e-02
               9.3147122e-01   2.8704671e-01   6.5959943e-02
               9.3769166e-01   2.7873707e-01   6.6168130e-02
               9.4389831e-01   2.7005145e-01   6.6398279e-02
               9.5009275e-01   2.6094683e-01   6.6650135e-02
               9.5627950e-01   2.5141834e-01   6.6923773e-02
               9.6245047e-01   2.4136103e-01   6.7218475e-02
               9.6861585e-01   2.3072982e-01   6.7533548e-02
               9.7476660e-01   2.1945144e-01   6.7869417e-02
               9.8091309e-01   2.0738932e-01   6.8229718e-02
               9.8705021e-01   1.9442270e-01   6.8619747e-02
               9.9317979e-01   1.8027042e-01   6.9032328e-02
               9.9929581e-01   1.6476261e-01   6.9441308e-02
               1.0000000e+00   1.4745190e-01   6.9829429e-02
               1.0000000e+00   1.2757129e-01   7.0234818e-02
               1.0000000e+00   1.0393299e-01   7.0711225e-02];
            
    end
         

 
end   

function colmap_demo

clf; 
axp=[.1 .1 .2 .08];
dx=[.29 0 0 0];
dy=[0 .19 0 0];

examples={...
    0,4,'m_colmap(''jet'',256)',      {'Perceptually uniform jet replacement','with diverging luminance'}
    1,4,'m_colmap(''blue'',256)',     'Good for bathymetry'
    2,4,'m_colmap(''land'',256)',     'Land from coastal wetlands to mountains'
    0,3,'m_colmap(''diverging'',256)',{'Currents, echo-sounder images','diverging luminance with a ''zero'''}
    1,3,'m_colmap(''water'',256)',    'Another bathymetry map'
    2,3,'m_colmap(''gland'',256)',    'Land with more green'
    0,2,'m_colmap(''odv'',256)',      'Isoluminant (add your own shading)'
    1,2,'m_colmap(''green'',256)',    'Chlorophyll? Land?'
    2,2,'m_colmap(''bland'',256)',    'Land without green'
    0,0,'m_colmap(''jet'',''step'',10)',{'Banded continuous map (256 colours)','sort of like contouring'}
    1,0,'m_colmap(''jet'',10)',        'A few discrete steps (10 colours)'
    2,0,'[m_colmap(''blues'',64);m_colmap(''gland'',128)]',{'Complex water + land example','must use ''caxis'' to get coastline correct'}
    };


for k=1:size(examples,1)
    ax(k)=axes('pos',axp+dx*examples{k,1}+dy*examples{k,2});
    imagesc(1:256);
    set(gca,'ytick',[],'xtick',[]);%[1 64:64:256]);
    eval(['colormap(ax(k),' examples{k,3} ');']);
    title([char('a'+k-1) ') ' examples{k,3}],'interp','none');
    xlabel(examples{k,4});
end

% print -dpng ./doc/exColmaps
end














